package com.bxm.lovelink.common.chat;

import cn.hutool.core.collection.ConcurrentHashSet;
import com.alibaba.fastjson.JSONObject;
import com.bxm.lovelink.common.chat.coze.CozeJWTOAuth;
import com.bxm.lovelink.common.chat.coze.CozeProperties;
import com.bxm.lovelink.common.chat.coze.MessageConvert;
import com.bxm.lovelink.common.chat.coze.MessageConvertFactory;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.ChatSession;
import com.bxm.lovelink.common.dal.entity.ChatSessionMessage;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.event.chat.ChatCompletionEvent;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.UUIDHelper;
import com.coze.openapi.client.chat.CancelChatReq;
import com.coze.openapi.client.chat.CreateChatReq;
import com.coze.openapi.client.chat.model.Chat;
import com.coze.openapi.client.chat.model.ChatEvent;
import com.coze.openapi.client.chat.model.ChatEventType;
import com.coze.openapi.client.connversations.message.model.Message;
import com.coze.openapi.client.connversations.message.model.MessageRole;
import com.coze.openapi.service.service.CozeAPI;
import com.google.common.collect.Maps;
import io.reactivex.functions.Consumer;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.ClientAbortException;
import org.apache.commons.lang3.StringUtils;

import java.io.InterruptedIOException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static com.coze.openapi.service.config.Consts.COZE_CN_BASE_URL;

/**
 * @author Allen Hu
 * @date 2025/6/26
 */
@Slf4j
public class CozeChatServiceImpl implements ChatService {

    private final CozeJWTOAuth cozeJWTOAuth;
    private final CozeProperties cozeProperties;
    private final EventPark eventPark;
    private final MessageConvertFactory messageConvertFactory;
    /**
     * 将要停止的会话
     */
    private final ConcurrentHashSet<String> stopping = new ConcurrentHashSet<>();

    public CozeChatServiceImpl(CozeJWTOAuth cozeJWTOAuth, CozeProperties cozeProperties, EventPark eventPark, MessageConvertFactory messageConvertFactory) {
        this.cozeJWTOAuth = cozeJWTOAuth;
        this.cozeProperties = cozeProperties;
        this.eventPark = eventPark;
        this.messageConvertFactory = messageConvertFactory;
    }

    @Override
    public void cancel(String conversionId) {
        stopping.add(conversionId);
    }

    private boolean isLogging(ChatRequest request) {
        return true;
    }

    @Override
    public void stream(ChatRequest request) {
        ChatSession chatSession = request.getChatSession();
        User user = request.getUser();
        List<ChatSessionMessage> chatSessionMessages = request.getChatSessionMessages();
        List<com.coze.openapi.client.connversations.message.model.Message> messageList = convert2Message(chatSessionMessages);

        final String conversionId = UUIDHelper.generate();
        final ChatStat chatStat = new ChatStat();
        final StringBuilder reasoningContentBuilder = new StringBuilder();
        final StringBuilder contentBuilder = new StringBuilder();

        String botId = cozeProperties.getBotId();
        String userId = "LOVELINK-" + user.getId();

        final CozeAPI coze = new CozeAPI.Builder()
                .baseURL(COZE_CN_BASE_URL)
                .auth(cozeJWTOAuth.build())
                .readTimeout((int) Duration.ofMinutes(5).toMillis())
                .build();
        try {
            // 发送建立连接的握手消息
            request.doWriteAndFlush(
                    SseEventData.builder().status(SseEventData.STATUS_CREATED).time(SseEventData.getNowTimeString()).conversionId(conversionId).build()
            );

            Map<String, String> varMaps = Maps.newHashMap();
            Map<String, Object> parameters = Optional.ofNullable(request.getParameter()).orElseGet(HashMap::new);

            CreateChatReq chatReq = CreateChatReq.builder()
                    .botID(botId)
                    .userID(userId)
                    .stream(true)
                    .conversationID(chatSession.getCozeConversationId())
                    .customVariables(varMaps)
                    .parameters(parameters)
                    .messages(messageList)
                    .build();

            if (isLogging(request)) {
                log.info("[COZE] {} - {}", conversionId, JSONObject.toJSONString(chatReq));
            }

            ChatEvent chatEvent = coze.chat().stream(chatReq)
                    .doOnRequest(t -> {
                        request.doWriteAndFlush(SseEventData.builder().status(SseEventData.STATUS_START).time(SseEventData.getNowTimeString()).conversionId(conversionId).build());
                    })
                    .doOnNext(new Consumer<ChatEvent>() {
                        @Override
                        public void accept(ChatEvent chatEvent) throws Exception {
                            ChatEventType event = chatEvent.getEvent();
                            Message message = chatEvent.getMessage();

                            this.setChatStatValues(chatStat, message);

                            boolean onEventChatInProgress = ChatEventType.CONVERSATION_CHAT_IN_PROGRESS.equals(event);
                            boolean onEventMessageDelta = ChatEventType.CONVERSATION_MESSAGE_DELTA.equals(event);

                            if (onEventChatInProgress || onEventMessageDelta) {
                                // 只允许这两个流程时可以打断
                                if (stopping.contains(conversionId)) {
                                    throw new InterruptedException("User canceled the conversation");
                                }
                            }

                            if (onEventMessageDelta) {
                                String reasoningContent = message.getReasoningContent();
                                String content = message.getContent();
                                if (StringUtils.isNotBlank(reasoningContent)) {
                                    reasoningContentBuilder.append(reasoningContent);
                                    request.doWriteAndFlush(SseEventData.builder().status(SseEventData.STATUS_MESSAGE).time(SseEventData.getNowTimeString()).conversionId(conversionId).reasoningContent(reasoningContent).build());
                                }
                                if (StringUtils.isNotBlank(content)) {
                                    contentBuilder.append(content);
                                    request.doWriteAndFlush(SseEventData.builder().status(SseEventData.STATUS_MESSAGE).time(SseEventData.getNowTimeString()).conversionId(conversionId).content(content).build());
                                }
                            } else if (ChatEventType.CONVERSATION_CHAT_FAILED.equals(event)) {
                                Chat chat = chatEvent.getChat();
                                throw new ChatFailedException(chat.getLastError().toString());
                            } else {
                                log.debug("other event: {} - {}", event.getValue(), message);
                            }
                        }

                        private void setChatStatValues(ChatStat chatStat, Message message) {
                            if (Objects.nonNull(message)) {
                                if (StringUtils.isNotBlank(message.getChatId())) {
                                    chatStat.setCozeChatId(message.getChatId());
                                }
                                if (StringUtils.isNotBlank(message.getBotId())) {
                                    chatStat.setCozeBotId(message.getBotId());
                                }
                                if (StringUtils.isNotBlank(message.getConversationId())) {
                                    chatStat.setCozeConversationId(message.getConversationId());
                                }
                            }
                        }
                    })
                    .lastElement()
                    .blockingGet();

            if (isLogging(request)) {
                log.info("[COZE] {} - Reasoning: {} | Content: {}", conversionId, reasoningContentBuilder, contentBuilder);
            }

        } catch (Exception e) {
            SseEventData errorSseData = SseEventData.builder()
                    .status(SseEventData.STATUS_FAILED)
                    .time(SseEventData.getNowTimeString())
                    .conversionId(conversionId)
                    .error(e.getMessage())
                    .build();

            if (e instanceof ChatFailedException) {
                // 接口返回明确的错误
                log.error("Chat Fail: {}", e.getMessage());
                request.doWriteAndFlushQuietly(errorSseData);
                throw new ChatException(e);
            }
            Throwable cause = e.getCause();
            if (cause instanceof InterruptedIOException) {
                // 网络异常
                request.doWriteAndFlushQuietly(errorSseData);
                throw new ChatFailedException("IO Error");
            } else if (cause instanceof ClientAbortException || cause instanceof InterruptedException) {
                // 用户主动断开了连接
                this.cancelQuietly(chatStat, coze);
            } else {
                log.error("call coze error: ", e);
                request.doWriteAndFlushQuietly(errorSseData);
                throw new ChatException(e);
            }
        } finally {
            // 移除正在停止的会话
            stopping.remove(conversionId);

            final ChatSessionMessage chatSessionMessage = new ChatSessionMessage()
                    .setSessionId(chatSession.getId())
                    .setRole(MessageRole.ASSISTANT.getValue())
                    .setContentType(Constants.Chat.CONTENT_TYPE_TEXT)
                    .setReasoningContent(reasoningContentBuilder.toString())
                    .setContent(contentBuilder.toString())
                    .setCreateTime(LocalDateTime.now());

            chatSessionMessages.add(chatSessionMessage);
            request.doComplete(chatSessionMessages);

            ChatCompletionEvent chatCompletionEvent = new ChatCompletionEvent(this,
                    request.getUser(),
                    chatSession,
                    chatSessionMessages,
                    reasoningContentBuilder.toString(),
                    contentBuilder.toString(),
                    chatStat);

            eventPark.post(chatCompletionEvent);
        }
    }

    private void cancelQuietly(ChatStat chatStat, CozeAPI coze) {
        try {
            CancelChatReq cancelChatReq = CancelChatReq.builder()
                    .chatID(chatStat.getCozeChatId())
                    .conversationID(chatStat.getCozeConversationId())
                    .build();
            coze.chat().cancel(cancelChatReq);
        } catch (Exception ignored) {
        }
    }

    private List<Message> convert2Message(List<ChatSessionMessage> messages) {
        return messages.stream().map(message -> {
            Integer contentType = message.getContentType();
            MessageConvert messageConvert = messageConvertFactory.get(contentType);
            if (messageConvert == null) {
                log.warn("未找到对应的消息转换器，contentType: {}", contentType);
                return null;
            }
            Message msg = messageConvert.convert(message);
            msg.setRole(MessageRole.fromString(message.getRole()));
            return msg;
        }).collect(Collectors.toList());
    }
}
