package com.bxm.lovelink.common.contant;

import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.collections.CollectionUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

/**
 * @author zhangdong
 * @date 2025/5/20
 */
public interface UserConstants {


    static <E extends Enum<E> & BaseEnum> String getNameByCode(Class<E> enumClass, Integer code) {
        if (code == null) {
            return null;
        }
        for (E e : enumClass.getEnumConstants()) {
            if (code.equals(e.getCode())) {
                return e.getName();
            }
        }
        return null;
    }

    static <E extends Enum<E> & BaseEnum> List<String> getNameByCode(Class<E> enumClass, List<Integer> codes) {
        List<String> names = new ArrayList<>();
        if (CollectionUtils.isEmpty(codes)) {
            return names;
        }
        for (E e : enumClass.getEnumConstants()) {
            if (codes.contains(e.getCode())) {
                names.add(e.getName());
            }
        }
        return names;
    }

    interface BaseEnum {

        Integer getCode();

        String getName();

        String getName2();

        Boolean getDisplayName();

        Boolean getDisplayName2();

    }


    @AllArgsConstructor
    @Getter
    enum EnumType {

        YEAR_REVENUE(1, YearRevenueEnum.values()),
        EDU_HIGHEST(2, EduHighestEnum.values()),
        MARRIAGE(3, MarriageStatusEnum.values()),
        CHILDREN(4, ChildrenStatusEnum.values()),
        LOVE_GOAL(5, LoveGoalEnum.values()),
        TATTOO_STATUS(6, TattooStatusEnum.values()),
        SMOKE_STATUS(7, SmokeStatusEnum.values()),
        HOUSE(8, HouseEnum.values()),
        CAR(9, CarEnum.values()),
        ZODIAC(10, UserZodiacEnum.values()),
        CONSTELLATION(11, UserConstellationEnum.values()),
        INDUSTRY(12, IndustryRangeEnum.values()),
        JOB(13, JobRangeEnum.values()),
        LIFE(14, LifeRangeEnum.values()),
        SCHOOL(15, SchoolRangeEnum.values()),
        CERT_TYPE(16, CertTypeEnum.values()),
        MARRIAGE_CHILDREN(17, MarriageChildrenStatusEnum.values()),
        MARRIAGE_CHILDREN_RELAX(18, MarriageChildrenRelaxStatusEnum.values()),
        BODY_SHAPE(19, BodyShapeEnum.values()),

        ;

        private final Integer type;
        private final BaseEnum[] baseEnums;
    }

    @AllArgsConstructor
    @Getter
    enum BodyShapeEnum implements BaseEnum {
        SLIM_ELEGANT(1, "纤瘦苗条型", "纤瘦苗条型", true, true),
        FIT_DYNAMIC(2, "健美活力型", "健美活力型", true, true),
        BALANCED_STANDARD(4, "匀称标准型", "匀称标准型", true, true),
        CURVY_HOURGLASS(8, "曲线沙漏型", "曲线沙漏型", true, true),
        PLUMP_FULL_FIGURE(16, "丰满丰腴型", "丰满丰腴型", true, true),

        LEAN_SLIM(1024, "纤瘦清瘦型", "纤瘦清瘦型", true, true),
        SPORTY_FIT(2048, "运动健美型", "运动健美型", true, true),
        MUSCULAR_BUILD(4096, "肌肉健硕型", "肌肉健硕型", true, true),
        BALANCED_STANDARD_AGAIN(8192, "匀称标准型", "匀称标准型", true, true),  // 与之前重复，可合并
        STRONG_STURDY(16384, "壮士敦厚型", "壮士敦厚型", true, true),


        ;


        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;


    }


    @AllArgsConstructor
    @Getter
    enum YearRevenueEnum implements BaseEnum {
        NO_PREFERENCE(1, "其他", "不介意", false, true),
        BELOW_5K(2, "5万以下", "5万以下", true, true),
        FROM_5K_TO_10K(4, "5-10万", "5-10万", true, true),
        FROM_10K_TO_20K(8, "10-20万", "10-20万", true, true),
        FROM_20K_TO_30K(16, "20-30万", "20-30万", true, true),
        FROM_30K_TO_40K(32, "30-40万", "30-40万", true, true),
        FROM_40K_TO_50K(64, "40-50万", "40-50万", true, true),
        FROM_50K_TO_100K(128, "50-100万", "50-100万", true, true),
        ABOVE_100K(256, "100万以上", "100万以上", true, true);


        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;


    }

    @AllArgsConstructor
    @Getter
    enum EduHighestEnum implements BaseEnum {
        NO_PREFERENCE(1, "其他", "其他", false, false),
        HIGH_SCHOOL_OR_BELOW(2, "高中及以下", "高中及以下", true, true),
        JUNIOR_COLLEGE(4, "大专", "大专", true, true),
        BACHELOR(8, "本科", "本科", true, true),
        MASTER(16, "硕士", "硕士", true, true),
        DOCTOR(32, "博士", "博士", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

    }

    @AllArgsConstructor
    @Getter
    enum MarriageChildrenStatusEnum implements BaseEnum {
        SINGLE(2, "未婚", "未婚", true, true),
        DIVORCED_NO_CHILDREN(4, "离异无子女", "离异无子女", true, true),
        DIVORCED_ONE_DAUGHTER(8, "离异有一个女儿", "离异有一个女儿", true, true),
        DIVORCED_ONE_SON(16, "离异有一个儿子", "离异有一个儿子", true, true),
        DIVORCED_MORE_CHILDREN(32, "离异有多个子女", "离异有多个子女", true, true),
        WIDOWED_NO_CHILDREN(64, "丧偶无子女", "丧偶无子女", true, true),
        WIDOWED_MORE_CHILDREN(128, "丧偶有子女", "丧偶有子女", true, true);


        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

    }

    @AllArgsConstructor
    @Getter
    enum MarriageChildrenRelaxStatusEnum implements BaseEnum {
        DIVORCED_NO_CHILDREN(4, "离异无子女", "离异无子女", true, true),
        DIVORCED_ONE_DAUGHTER(8, "离异有一个女儿", "离异有一个女儿", true, true),
        DIVORCED_ONE_SON(16, "离异有一个儿子", "离异有一个儿子", true, true),
        DIVORCED_MORE_CHILDREN(32, "离异有多个子女", "离异有多个子女", true, true),
        WIDOWED_NO_CHILDREN(64, "丧偶无子女", "丧偶无子女", true, true),
        WIDOWED_MORE_CHILDREN(128, "丧偶有子女", "丧偶有子女", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

    }

    @AllArgsConstructor
    @Getter
    enum MarriageStatusEnum implements BaseEnum {

        NO_PREFERENCE(1, "不限", "不限", false, true),
        SINGLE(2, "未婚", "未婚", true, true),
        DIVORCED(4, "离异", "离异", true, true),
        WIDOWED(8, "丧偶", "丧偶", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

    }

    @AllArgsConstructor
    @Getter
    enum ChildrenStatusEnum implements BaseEnum {
        NO_PREFERENCE(1, "无", "其他", true, true),
        NO_CHILDREN(2, "无子女", "无子女", false, true),
        ONE_SON(4, "有一个儿子", "一个儿子", true, true),
        ONE_DAUGHTER(8, "有一个女儿", "一个女女", true, true),
        ONE_SON_ONE_DAUGHTER(16, "有一子一女", "一儿一女", true, true),
        TWO_SONS(32, "有两个儿子", "两个儿子", true, true),
        TWO_DAUGHTERS(64, "有两个女儿", "两个女儿", true, true),
        MORE_THAN_TWO(128, "有两个以上子女", "两个以上子女", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }


    @AllArgsConstructor
    @Getter
    enum LoveGoalEnum implements BaseEnum {
        UNDECIDED(1, "没考虑清楚", "没考虑清楚", true, true),
        WANT_MARRIAGE_SOON(2, "短期内想结婚", "短期内想结婚", true, true),
        SERIOUS_DATING(4, "先认真谈场恋爱", "先认真谈场恋爱", true, true),
        DATING_THEN_MAY_MARRY(8, "先认真谈场恋爱，合适可以结婚", "先认真谈场恋爱，合适可以结婚", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum TattooStatusEnum implements BaseEnum {
        NO_PREFERENCE(1, "无所谓", "无所谓/不介意", false, true),
        SMALL_OR_HIDDEN(2, "小面积/隐藏部位纹身", "可接受小面积/隐藏部位纹身", true, true),
        LARGE_OR_VISIBLE(4, "大面积/明显部位纹身", "可接受大面积/明显部位纹身", true, true),
        NO_TATTOO(8, "没纹身", "不接受任何纹身", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum CertTypeEnum implements BaseEnum {

        REAL_PEOPLE(1, "实名认证", "实名认证", true, true),
        EDU(2, "学历认证", "学历认证", true, true),
        MARRIED(4, "婚姻认证", "婚姻认证", true, true),
        HOUSE(8, "房子认证", "房子认证", true, true),
        CAR(16, "车认证", "车认证", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

        public static CertTypeEnum getCertType(Integer code) {
            if (code == null) {
                return null;
            }
            for (CertTypeEnum certTypeEnum : CertTypeEnum.values()) {
                if (certTypeEnum.getCode().equals(code)) {
                    return certTypeEnum;
                }
            }
            return null;
        }

        /**
         * 获取需要的认证类型
         * @return
         */
        public static List<Integer> getNeededCertTypes() {
            return Lists.newArrayList(REAL_PEOPLE.getCode(), EDU.getCode(), MARRIED.getCode(), CAR.getCode());
        }
    }

    @AllArgsConstructor
    @Getter
    enum SmokeStatusEnum implements BaseEnum {
        NO_PREFERENCE(1, "无所谓", "无所谓/不介意", false, true),
        OCCASIONAL(2, "偶尔抽烟", "可接受偶尔抽烟", true, true),
        REGULAR(4, "抽烟", "可接受抽烟", true, true),
        NON_SMOKER(8, "不抽烟", "不接受抽烟", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

    }

    @AllArgsConstructor
    @Getter
    enum HouseEnum implements BaseEnum {
        NO_PREFERENCE(1, "无所谓", "无所谓", false, true),
        HAS_HOUSE(2, "已购房", "希望对方有房", true, true),
        NO_HOUSE(4, "暂无", "接受对方没有房", true, true);


        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;

    }

    @AllArgsConstructor
    @Getter
    enum CarEnum implements BaseEnum {
        NO_PREFERENCE(1, "无所谓", "无所谓", false, true),
        HAS_CAR(2, "已购车", "希望对方有车", true, true),
        NO_CAR(4, "暂无", "接受对方没有车", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum IndustryRangeEnum implements BaseEnum {

        NO_PREFERENCE(1, "无所谓", "无所谓", true, true),
        SAME_INDUSTRY(2, "同行", "同行", true, true),
        DIFFERENT_INDUSTRY(4, "非同行", "非同行", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum JobRangeEnum implements BaseEnum {

        NO_PREFERENCE(1, "无所谓", "无所谓", true, true),
        STABLE_JOB(2, "稳定工作", "稳定工作", true, true),
        ENTREPRENEURSHIP(4, "创业经商", "创业经商", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum LifeRangeEnum implements BaseEnum {

        NO_PREFERENCE(1, "无所谓", "无所谓", true, true),
        FAMILY_ORIENTED(2, "顾家型", "顾家型", true, true),
        CAREER_ORIENTED(4, "事业型", "事业型", true, true);

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum SchoolRangeEnum implements BaseEnum {

        NO_LIMIT(1, "不限", "不限", false, false),
        TOP_SCHOOLS(2, "985、211院校", "985、211院校", true, true);
        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
    }

    @AllArgsConstructor
    @Getter
    enum UserZodiacEnum implements BaseEnum {
        RAT(0, "鼠", "鼠", true, true),
        OX(1, "牛", "牛", true, true),
        TIGER(2, "虎", "虎", true, true),
        RABBIT(3, "兔", "兔", true, true),
        DRAGON(4, "龙", "龙", true, true),
        SNAKE(5, "蛇", "蛇", true, true),
        HORSE(6, "马", "马", true, true),
        GOAT(7, "羊", "羊", true, true),
        MONKEY(8, "猴", "猴", true, true),
        ROOSTER(9, "鸡", "鸡", true, true),
        DOG(10, "狗", "狗", true, true),
        PIG(11, "猪", "猪", true, true);


        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;


        public static UserZodiacEnum getByBirthday(LocalDate birthday) {
            int zodiac = ((birthday.getYear() - 2020) % 12 + 12) % 12;
            return UserZodiacEnum.values()[zodiac];
        }
    }

    @AllArgsConstructor
    @Getter
    enum UserConstellationEnum implements BaseEnum {

        ARIES(0, "白羊座", "白羊座", true, true, date -> isBetween(date, 3, 21, 4, 19)),
        TAURUS(1, "金牛座", "金牛座", true, true, date -> isBetween(date, 4, 20, 5, 20)),
        GEMINI(2, "双子座", "双子座", true, true, date -> isBetween(date, 5, 21, 6, 20)),
        CANCER(3, "巨蟹座", "巨蟹座", true, true, date -> isBetween(date, 6, 21, 7, 22)),
        LEO(4, "狮子座", "狮子座", true, true, date -> isBetween(date, 7, 23, 8, 22)),
        VIRGO(5, "处女座", "处女座", true, true, date -> isBetween(date, 8, 23, 9, 22)),
        LIBRA(6, "天秤座", "天秤座", true, true, date -> isBetween(date, 9, 23, 10, 23)),
        SCORPIO(7, "天蝎座", "天蝎座", true, true, date -> isBetween(date, 10, 24, 11, 22)),
        SAGITTARIUS(8, "射手座", "射手座", true, true, date -> isBetween(date, 11, 23, 12, 21)),
        CAPRICORN(9, "摩羯座", "摩羯座", true, true, date -> isBetween(date, 12, 22, 1, 19)),
        AQUARIUS(10, "水瓶座", "水瓶座", true, true, date -> isBetween(date, 1, 20, 2, 18)),
        PISCES(11, "双鱼座", "双鱼座", true, true, date -> isBetween(date, 2, 19, 3, 20));

        private final Integer code;
        private final String name;
        private final String name2;
        private final Boolean displayName;
        private final Boolean displayName2;
        private final Function<LocalDate, Boolean> validator;

        public static UserConstellationEnum getByBirthday(LocalDate date) {
            for (UserConstellationEnum userConstellationEnum : UserConstellationEnum.values()) {
                if (userConstellationEnum.validator.apply(date)) {
                    return userConstellationEnum;
                }
            }
            return null;
        }

        public static UserConstellationEnum getByCode(Integer code) {
            if (code == null) {
                return null;
            }
            for (UserConstellationEnum userConstellationEnum : UserConstellationEnum.values()) {
                if (userConstellationEnum.getCode().equals(code)) {
                    return userConstellationEnum;
                }
            }
            return null;
        }


        private static boolean isBetween(LocalDate date, int startMonth, int startDay, int endMonth, int endDay) {
            LocalDate start = LocalDate.of(date.getYear(), startMonth, startDay);
            LocalDate end = LocalDate.of(date.getYear(), endMonth, endDay);

            if (end.isBefore(start)) {
                end = end.plusYears(1);
            }

            if (date.isBefore(start) && end.getMonthValue() == 1) {
                start = start.minusYears(1);
                end = end.minusYears(1);
            }

            return !date.isBefore(start) && !date.isAfter(end);
        }
    }
}