package com.bxm.lovelink.common.dal.manager;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.chat.*;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.ChatSession;
import com.bxm.lovelink.common.dal.entity.ChatSessionMessage;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.entity.dto.chat.ApiChatDto;
import com.bxm.lovelink.common.dal.manager.parameter.ParameterBuilderFactory;
import com.bxm.lovelink.common.dal.manager.parameter.ParameterBuilderRequest;
import com.bxm.lovelink.common.dal.manager.welcome.WelcomeBuilderFactory;
import com.bxm.lovelink.common.dal.manager.welcome.WelcomeBuilderRequest;
import com.bxm.lovelink.common.dal.service.IChatSessionMessageService;
import com.bxm.lovelink.common.dal.service.IChatSessionService;
import com.bxm.lovelink.common.dal.service.IUserService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.ClientAbortException;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.function.BiConsumer;

/**
 * @author Allen Hu
 * @date 2025/6/30
 */
@Slf4j
@Component
public class ApiChatManager {

    private final IUserService userService;
    private final IChatSessionService chatSessionService;
    private final IChatSessionMessageService chatSessionMessageService;
    private final ChatService chatService;
    private final ParameterBuilderFactory parameterBuilderFactory;
    private final WelcomeBuilderFactory welcomeBuilderFactory;

    public ApiChatManager(IUserService userService, IChatSessionService chatSessionService, IChatSessionMessageService chatSessionMessageService, ChatService chatService, ParameterBuilderFactory parameterBuilderFactory, WelcomeBuilderFactory welcomeBuilderFactory) {
        this.userService = userService;
        this.chatSessionService = chatSessionService;
        this.chatSessionMessageService = chatSessionMessageService;
        this.chatService = chatService;
        this.parameterBuilderFactory = parameterBuilderFactory;
        this.welcomeBuilderFactory = welcomeBuilderFactory;
    }

    public List<String> buildWelcome(User user) {
        WelcomeBuilderRequest request = WelcomeBuilderRequest
                .builder()
                .user(user)
                .build();
        return welcomeBuilderFactory.build(request);
    }

    public ChatSession createSessionIfNecessary(Long userId) {
        LambdaQueryWrapper<ChatSession> queryWrapper = Wrappers.lambdaQuery(ChatSession.class)
                .eq(ChatSession::getUserId, userId)
                .eq(ChatSession::getSceneType, Constants.Chat.SCENE_TYPE_DEFAULT);
        ChatSession chatSession = chatSessionService.getOne(queryWrapper);
        if (null == chatSession) {
            chatSession = new ChatSession();
            chatSession.setUserId(userId);
            this.chatSessionService.save(chatSession);
        }
        return chatSession;
    }

    public IPage<ChatSessionMessage> page(Long userId, int pageNum, int pageSize) {
        ChatSession chatSession = createSessionIfNecessary(userId);
        return chatSessionMessageService.page(new Page<>(pageNum, pageSize),
                new LambdaQueryWrapper<ChatSessionMessage>()
                        .eq(ChatSessionMessage::getSessionId, chatSession.getId())
                        .orderByDesc(ChatSessionMessage::getId));
    }

    public void streamChat(User user, ApiChatDto apiChatDto,
                           ByteWriter<byte[]> writeConsumer,
                           SimpleHandler flusher,
                           CompleteConsumer completeConsumer,
                           BiConsumer<ClientAbortException, String> clientAbortExceptionStringBiConsumer) {
        ChatSession session = createSessionIfNecessary(user.getId());

        // 保存用户的消息
        ChatSessionMessage message = apiChatDto.to()
                .setSessionId(session.getId())
                .setRole(Constants.Chat.ROLE_USER);
        chatSessionMessageService.save(message);

        ChatRequest chatRequest = ChatRequest.builder()
                .user(user)
                .chatSession(session)
                .chatSessionMessages(Lists.newArrayList(
                        message
                ))
                .parameter(parameterBuilderFactory.build(
                        ParameterBuilderRequest.builder()
                                .userId(user.getId())
                                .chatSession(session)
                                .apiChatDto(apiChatDto)
                                .me(userService.getCompositeById(user.getId()))
                                .ta(userService.getCompositeById(apiChatDto.getTaUserId()))
                                .build()
                ))
                .writeConsumer(writeConsumer)
                .flusher(flusher)
                .complete(messages -> {
                    ChatSessionMessage lastMessage = messages.get(messages.size() - 1);
                    chatSessionMessageService.save(lastMessage);
                    log.debug("save lastMessage: {}", lastMessage);
                    if (null != completeConsumer) {
                        completeConsumer.accept(messages);
                    }
                })
                .clientAbortExceptionStringBiConsumer(clientAbortExceptionStringBiConsumer)
                .build();
        chatService.stream(chatRequest);
    }

    public void cancel(String conversionId) {
        chatService.cancel(conversionId);
    }

}
