package com.bxm.lovelink.common.dal.manager;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.chat.ChatRequest;
import com.bxm.lovelink.common.chat.ChatService;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.lovelink.common.dal.entity.ChatSession;
import com.bxm.lovelink.common.dal.entity.ChatSessionMessage;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.entity.UserOtherInfo;
import com.bxm.lovelink.common.dal.entity.dao.ChatSessionDao;
import com.bxm.lovelink.common.dal.entity.dto.chat.ChatDto;
import com.bxm.lovelink.common.dal.entity.dto.chat.SessionMessageQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.chat.ChatSessionMessageVo;
import com.bxm.lovelink.common.dal.entity.vo.chat.ChatSessionVo;
import com.bxm.lovelink.common.dal.mapping.ChatSessionMapping;
import com.bxm.lovelink.common.dal.mapping.ChatSessionMessageMapping;
import com.bxm.lovelink.common.dal.service.IChatSessionMessageFeedbackService;
import com.bxm.lovelink.common.dal.service.IChatSessionMessageService;
import com.bxm.lovelink.common.dal.service.IChatSessionService;
import com.bxm.lovelink.common.dal.service.IUserOtherInfoService;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.stereotype.Service;

import java.io.OutputStream;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author Allen Hu
 * @date 2025/6/26
 */
@Slf4j
@Service
@AllArgsConstructor
public class ChatManager {

    private final IChatSessionService chatSessionService;
    private final IChatSessionMessageService chatSessionMessageService;
    private final IChatSessionMessageFeedbackService chatSessionMessageFeedbackService;
    private final IUserOtherInfoService userOtherInfoService;
    private final ChatService chatService;
    private final Fetcher fetcher;
    private final Counter counter;
    private final Updater updater;


    public ChatSession getSession(Long userId, Long sessionId) {
        // 可扩展成切面实现
        QueryWrapper<ChatSession> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(ChatSession.ID, sessionId);
        queryWrapper.eq(ChatSession.USER_ID, userId);
        ChatSession chatSession = chatSessionService.getOne(queryWrapper);
        if (null == chatSession) {
            throw new IllegalArgumentException("会话不存在");
        }
        return chatSession;
    }

    public List<ChatSessionVo> listSession(Long userId, Integer chatContactRole) {
        List<ChatSessionDao> list = chatSessionService.listDao(new LambdaQueryWrapper<ChatSession>()
                .last(chatContactRole != null, String.format("where chat_contact_role = %s order by last_message_time desc limit 100", chatContactRole)));

        list.forEach(session -> {
            Integer unreadNum = countUnreadNum(userId, session.getId());
            session.setUnreadNum(Optional.ofNullable(unreadNum).orElse(0));
        });

        list.sort(Comparator
                .comparing((ChatSessionDao o) -> o.getUnreadNum() == 0));

        return ChatSessionMapping.INSTANCE.toVosDyDao(list);
    }

    public IPage<ChatSessionMessageVo> listSessionMessage(SessionMessageQueryDto dto) {
        IPage<ChatSessionMessage> page = new Page<ChatSessionMessage>()
                .setCurrent(dto.getCurrent())
                .setSize(dto.getSize());
        LambdaQueryWrapper<ChatSessionMessage> query = new LambdaQueryWrapper<ChatSessionMessage>()
                .eq(ChatSessionMessage::getSessionId, dto.getSessionId())
                .orderByDesc(ChatSessionMessage::getId);
        IPage<ChatSessionMessage> result = chatSessionMessageService.page(page, query);
        return ChatSessionMessageMapping.INSTANCE.toPage(result);
    }

    public Integer countUnreadNum(Long userId, Long sessionId) {
        return fetcher.hfetch(RedisKeys.hashChatUnreadNum(sessionId), userId.toString(), Integer.class);
    }

    /**
     * 未读数+1
     * @param userId
     * @param sessionId
     */
    public void incrUnreadNum(Long userId, Long sessionId) {
        counter.hincrementAndGet(RedisKeys.hashChatUnreadNum(sessionId), Objects.toString(userId), RedisKeys.expireTimeOneYear);
    }

    /**
     * 消除未读数
     * @param userId 如果是员工传 0
     * @param sessionId
     */
    public void readMessage(Long userId, Long sessionId) {
        updater.hremove(RedisKeys.hashChatUnreadNum(sessionId), userId.toString());
    }

    public void switchChatRole(Long sessionId, Integer chatContactRole) {
        ChatSession session = chatSessionService.getById(sessionId);
        Long userId = session.getUserId();
        userOtherInfoService.update(new LambdaUpdateWrapper<UserOtherInfo>()
                .eq(UserOtherInfo::getUserId, userId)
                .set(UserOtherInfo::getChatContactRole, chatContactRole));
    }

    public void sendMessage(ChatDto dto) {
        ChatSession chatSession = chatSessionService.getById(dto.getSessionId());
        if (chatSession == null) {
            throw new IllegalStateException("会话不存在");
        }
        // 保存消息
        ChatSessionMessage message = ChatSessionMessageMapping.INSTANCE.dtoToDo(dto);
        chatSessionMessageService.save(message);
        // 用户未读+1
        Long userId = dto.getRole().equals(Constants.Chat.ROLE_STAFF) ? chatSession.getUserId() : Constants.Chat.STAFF_USER_ID;
        incrUnreadNum(message.getSessionId(), userId);
    }

    public void aiStream(ChatDto chatDto, OutputStream outputStream) {
        ChatSession session = chatSessionService.getById(chatDto.getSessionId());

        ChatRequest chatRequest = ChatRequest.builder()
                .user(new User().setId(Constants.Chat.STAFF_USER_ID))
                .chatSession(session)
                .chatSessionMessages(Lists.newArrayList(
                        chatDto.to()
                ))
                .writeConsumer(outputStream::write)
                .flusher(outputStream::flush)
                .complete(messages -> {
                    ChatSessionMessage lastMessage = messages.get(messages.size() - 1);
                    log.debug("ai message: {}", lastMessage);
                })
                .clientAbortExceptionStringBiConsumer((e, s) -> IOUtils.closeQuietly(outputStream))
                .build();
        chatService.stream(chatRequest);
    }

}
