package com.bxm.lovelink.common.dal.manager;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.contant.UserConstants;
import com.bxm.lovelink.common.dal.entity.*;
import com.bxm.lovelink.common.dal.entity.dao.MeetGroupDao;
import com.bxm.lovelink.common.dal.entity.dto.meet.*;
import com.bxm.lovelink.common.dal.entity.dto.meetticket.ReturnTicketOperateDto;
import com.bxm.lovelink.common.dal.entity.dto.meetticket.TicketOperateDto;
import com.bxm.lovelink.common.dal.service.*;
import com.bxm.lovelink.common.enums.TicketRecordTypeEnum;
import com.bxm.lovelink.common.enums.TicketReturnTypeEnum;
import com.bxm.lovelink.common.event.meet.*;
import com.bxm.lovelink.common.integration.sensitive.SensitiveTextResult;
import com.bxm.lovelink.common.push.Pusher;
import com.bxm.lovelink.common.push.factory.PushScene;
import com.bxm.lovelink.common.push.factory.PushTarget;
import com.bxm.lovelink.common.utils.UserUtils;
import com.bxm.lovelink.integration.ai.TextCompletionFacadeIntegration;
import com.bxm.warcar.integration.eventbus.EventPark;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 见面群相关功能
 *
 * @author Allen Hu
 * @date 2025/5/15
 */
@Slf4j
@Service
@AllArgsConstructor
public class MeetGroupManager {

    private final IMeetGroupService meetGroupService;
    private final IMeetGroupMessageService meetGroupMessageService;
    private final IUserService userService;
    private final IUserBasicInfoService userBasicInfoService;
    private final IUserMeetTicketService userMeetTicketService;
    private final IUserMeetTicketRecordService userMeetTicketRecordService;
    private final IActivityUserMutualSelectionService activityUserMutualSelectionService;
    private final ISensitiveService sensitiveService;
    private final VirtualPhoneNumberService virtualPhoneNumberService;
    private final EventPark eventPark;
    private final Pusher pusher;
    private final TextCompletionFacadeIntegration textCompletionFacadeIntegration;

    public MeetGroupDao findMeetGroupById(Long meetGroupId) {
        return meetGroupService.getDaoById(meetGroupId);
    }

    public MeetGroupDao findMeetGroupById(Long inviteUserId, Long acceptUserId) {
        QueryWrapper<MeetGroup> queryWrapper = new QueryWrapper<MeetGroup>()
                .and(meetGroupQueryWrapper -> meetGroupQueryWrapper
                        .eq(MeetGroup.INVITE_USER_ID, inviteUserId)
                        .eq(MeetGroup.ACCEPT_USER_ID, acceptUserId))
                .or(meetGroupQueryWrapper -> meetGroupQueryWrapper
                        .eq(MeetGroup.INVITE_USER_ID, acceptUserId)
                        .eq(MeetGroup.ACCEPT_USER_ID, inviteUserId))
                ;
        return meetGroupService.getDaoOne(queryWrapper);
    }

    /**
     * 生成邀请函信息
     * @param acceptUserId 受邀用户ID
     * @return 文案
     */
    public String generateInviteMessage(Long acceptUserId) {
        UserComposite user = userService.getCompositeById(acceptUserId);
        if (user == null) {
            throw new IllegalArgumentException("无效的用户");
        }
        UserBasicInfo userBasicInfo = user.getUserBasicInfo();

        String system = String.format(
                "## 角色定义\n" +
                "你是一个专业的文案大师，请基于我提供的信息，写一段邀请对方见面约会的话，不能提及约会地点，主要表达欣赏对方想进一步认识一下。\n" +
                "根据对方的信息来写出合适的内容，要求有一定水平和诗意，开头最好加上对方的称呼，要求控制在50个字以内。\n" +
                "\n" +
                "## 对方的信息\n" +
                "%s",
                userBasicInfo
        );
        String query = "请写一段";
        return textCompletionFacadeIntegration.chatInviteMessage(system, query);
    }

    /**
     * 发起见面
     *
     * @param requestMeetInviteDto 请求见面的参数
     * @return 建立的见面群信息
     */
    @Transactional(rollbackFor = Exception.class)
    public MeetGroup requestMeetInvite(Long inviteUserId, RequestMeetInviteDto requestMeetInviteDto) {
        // 发起见面：
        // 首先检查用户与对方的见面群是否不存在，不存在才能继续
        // 然后检查用户选择的相亲券支付情况，检查自己是否VIP，否则扣除相应的相亲券数量
        // 创建见面群，设置状态后返回
        Long acceptUserId = requestMeetInviteDto.getUserId();
        MeetGroupDao meetGroup = findMeetGroupById(inviteUserId, acceptUserId);

        boolean isFreeRequest = requestMeetInviteDto.isFreeRequest();
        int meetInviteNeedTicketCount = requestMeetInviteDto.getMeetInviteNeedTicketCount();
        int meetAcceptNeedTicketCount = requestMeetInviteDto.getMeetAcceptNeedTicketCount();

        if (Objects.nonNull(meetGroup)) {
            boolean accepted = meetGroup.getGroupStatus() >= Constants.MeetGroupConst.GroupStatus.ACCEPT.value();
            if (accepted) {
                if (isFreeRequest) {
                    // 如果免费邀约，重复请求那么就不做任何处理，直接返回
                    return meetGroup;
                }
                // 已经进入见面流程了
                throw new IllegalStateException("当前见面流程，不需要邀约");
            }
            boolean timeout = Objects.equals(meetGroup.getGroupStatus(), Constants.MeetGroupConst.GroupStatus.TIMEOUT.value());
            if (!isFreeRequest && !timeout) {
                // 如果不是免费邀约，也非超时的
                throw new IllegalStateException("对方已拒绝，无法重复邀约");
            }
        }

        int meetNeedTicketCount = Constants.MeetGroupConst.MEET_NEED_TICKET_COUNT;

        log.info("发起见面：发起人：{}，受邀人：{}，需要消耗的红豆：{}，发起人支出：{}，受邀人支出：{}",
                inviteUserId, acceptUserId, meetNeedTicketCount, meetInviteNeedTicketCount, meetAcceptNeedTicketCount);

        // 找出女生，未来可能是被邀请方
        Map<Long, UserBasicInfo> userBasicInfoMap = getUserBasicInfoMap(inviteUserId, acceptUserId);
        UserBasicInfo inviteUser = userBasicInfoMap.get(inviteUserId);
        UserBasicInfo acceptUser = userBasicInfoMap.get(acceptUserId);

        UserBasicInfo female = filterFemale(userBasicInfoMap);
        if (null == female) {
            throw new IllegalStateException("sorry，没有女生，无法继续发起");
        }
        UserBasicInfo male = filterMale(userBasicInfoMap);
        if (null == male) {
            throw new IllegalStateException("sorry，没有男生，无法继续发起");
        }

        if (isFreeRequest) {
            // 如果不需要支付红豆，那说明是免费邀约。
            boolean exists = activityUserMutualSelectionService.exists(male.getUserId(), female.getUserId());
            if (!exists) {
                throw new IllegalStateException("免费邀约时，必须是互选关系");
            }
            // 设置为0
            meetNeedTicketCount = 0;
        } else {
            // 查询自己是不是VIP
            if (UserUtils.isVip(inviteUser.getVipExpireTime())) {
                meetNeedTicketCount -= Constants.MeetGroupConst.VIP_REDUCTION_TICKET_COUNT;
                meetInviteNeedTicketCount -= Constants.MeetGroupConst.VIP_REDUCTION_TICKET_COUNT;
            }

            // 如果对方也是VIP，就不需要扣
            if (UserUtils.isVip(acceptUser.getVipExpireTime())) {
                meetNeedTicketCount -= Constants.MeetGroupConst.VIP_REDUCTION_TICKET_COUNT;
                meetAcceptNeedTicketCount -= Constants.MeetGroupConst.VIP_REDUCTION_TICKET_COUNT;
            }
        }

        log.info("发起见面经过VIP计算后：发起人：{}，受邀人：{}，需要消耗的相亲券：{}，发起人支出：{}，受邀人支出：{}",
                inviteUserId, acceptUserId, meetNeedTicketCount, meetInviteNeedTicketCount, meetAcceptNeedTicketCount);

        if (meetNeedTicketCount - meetInviteNeedTicketCount - meetAcceptNeedTicketCount != 0) {
            throw new IllegalStateException("发起见面的相亲券数量无效");
        }

        String inviteMessage = requestMeetInviteDto.getInviteMessage();
        SensitiveTextResult sensitiveTextResult = sensitiveService.checkText(inviteMessage);
        if (sensitiveTextResult.getHit()) {
            log.warn("sensitive text '{}' on message: {}", sensitiveTextResult.getText(), inviteMessage);
            throw new IllegalArgumentException("邀请函内容不符合要求");
        }

        MeetGroup e = new MeetGroup()
                .setInviteUserId(inviteUserId)
                .setAcceptUserId(acceptUserId)
                .setInviteMessage(inviteMessage)
                .setInviteUserReadMark(Constants.YES)
                .setAcceptUserReadMark(Constants.NO)
                .setGroupStatus(Constants.MeetGroupConst.GroupStatus.WAIT_ACCEPT.value())
                .setMeetNeedTicketCount(meetNeedTicketCount)
                .setMeetInviteNeedTicketCount(meetInviteNeedTicketCount)
                .setMeetAcceptNeedTicketCount(meetAcceptNeedTicketCount)
                .setMeetInviteUseTicketCount(0)
                .setMeetAcceptUseTicketCount(0)
                .setMeetSelectAddressUserId(female.getUserId())
                .setMeetStatus(Constants.MeetGroupConst.MeetAddressStatus.WAIT_SELECT.value())
                .setInviteExpireTime(LocalDateTime.now().plusHours(Constants.MeetGroupConst.INVITE_EXPIRE_TIME_HOUR).withSecond(0));
        if (Objects.nonNull(meetGroup)) {
            e.setId(meetGroup.getId());
        }
        meetGroupService.saveOrUpdate(e);

        // 处理邀请方消耗的红豆
        if (meetInviteNeedTicketCount > 0) {
            TicketOperateDto dto = new TicketOperateDto()
                    .setUserId(inviteUserId)
                    .setConnectId(e.getId())
                    .setAmount(meetInviteNeedTicketCount)
                    .setRemark(String.format("邀约%s", acceptUser.getNickname()));
            userMeetTicketService.useTicket(dto);
            log.info("用户：{}，消耗了：{}", inviteUserId, meetInviteNeedTicketCount);
            e.setMeetInviteUseTicketCount(meetInviteNeedTicketCount);
        }
        meetGroupService.saveOrUpdate(e);


        PushTarget pushTarget =new PushTarget();
        pushTarget.setUserId(acceptUserId);
        pushTarget.addExtend("nickName",inviteUser.getNickname());
        pusher.doImmediatePush(PushScene.MEET_REQUEST, pushTarget);
        eventPark.post(new MeetGroupRequestEvent(this, e, inviteUserId));

        if (isFreeRequest) {
            // 如果是免费邀约的，那么被邀请方自动同意
            this.acceptMeetInvite(acceptUserId, e.getId());
        }
        return e;
    }

    /**
     * 查询见面群
     *
     * @param userId 用户id
     * @param meetGroupId 见面群id
     * @return 见面群信息
     */
    public MeetGroupDao queryMeetInvite(Long userId, Long meetGroupId) {
        MeetGroupDao meetGroup = checkGroupInviteOrAcceptIsMine(meetGroupId, userId);
        // 标记我的已读状态
        if (meetGroup.getInviteUserId().equals(userId)) {
            meetGroup.setInviteUserReadMark(Constants.YES);
        }
        if (meetGroup.getAcceptUserId().equals(userId)) {
            meetGroup.setAcceptUserReadMark(Constants.YES);
        }
        meetGroupService.updateById(meetGroup);
        return meetGroup;
    }

    @Transactional(rollbackFor = Exception.class)
    public MeetGroupDao acceptMeetInvite(Long acceptUserId, Long meetGroupId) {
        MeetGroupDao meetGroup = checkGroupAcceptIsMe(meetGroupId, acceptUserId);
        if (meetGroup.getGroupStatus() != Constants.MeetGroupConst.GroupStatus.WAIT_ACCEPT.value()) {
            throw new IllegalStateException("见面群状态无效");
        }
        if (meetGroup.isExpiredInInviteTimeOut()) {
            throw new IllegalStateException("见面群邀请已过期");
        }
        meetGroup.setGroupStatus(Constants.MeetGroupConst.GroupStatus.ACCEPT.value());
        meetGroup.setInviteUserReadMark(Constants.YES);
        meetGroup.setAcceptUserReadMark(Constants.YES);
        // 支付相亲券
        Integer meetAcceptNeedTicketCount = meetGroup.getMeetAcceptNeedTicketCount();
        if (meetAcceptNeedTicketCount > 0) {
            // 判断用户是否VIP
            UserBasicInfo acceptUserBasicInfo = meetGroup.getAcceptUserBasicInfo();
            if (UserUtils.isVip(acceptUserBasicInfo.getVipExpireTime())) {
                meetAcceptNeedTicketCount -= Constants.MeetGroupConst.VIP_REDUCTION_TICKET_COUNT;
                log.info("接受见面：受邀人是VIP，需要支出：{}", meetAcceptNeedTicketCount);
            }
            log.info("接受见面：受邀人需要支出：{}", meetAcceptNeedTicketCount);
            if (meetAcceptNeedTicketCount > 0) {
                TicketOperateDto dto = new TicketOperateDto()
                        .setUserId(acceptUserId)
                        .setConnectId(meetGroupId)
                        .setAmount(meetAcceptNeedTicketCount);
                userMeetTicketService.useTicket(dto);
                log.info("用户：{}，消耗了：{}", acceptUserId, meetAcceptNeedTicketCount);
            }

            meetGroup.setMeetAcceptUseTicketCount(meetAcceptNeedTicketCount);
        }
        meetGroupService.updateById(meetGroup);

        eventPark.post(new MeetGroupAcceptedEvent(this, meetGroup, acceptUserId));

        PushTarget pushTarget =new PushTarget();
        pushTarget.setUserId(meetGroup.getInviteUserId());
        pushTarget.addExtend("nickName",meetGroup.getAcceptUserBasicInfo().getNickname());
        pusher.doImmediatePush(PushScene.ACCEPT_MEET_REQUEST, pushTarget);
        return meetGroup;
    }

    @Transactional(rollbackFor = Exception.class)
    public MeetGroupDao rejectMeetInvite(Long acceptUserId, Long meetGroupId, String reason) {
        MeetGroupDao meetGroup = checkGroupAcceptIsMe(meetGroupId, acceptUserId);
        if (meetGroup.getGroupStatus() != Constants.MeetGroupConst.GroupStatus.WAIT_ACCEPT.value()) {
            throw new IllegalStateException("见面群状态无效");
        }
        if (meetGroup.isExpiredInInviteTimeOut()) {
            throw new IllegalStateException("见面群邀请已过期");
        }
        this.refundTicket(meetGroupId, meetGroup.getInviteUserId(), TicketReturnTypeEnum.REFUSE);
        meetGroup.setMeetInviteUseTicketCount(0);
        meetGroup.setGroupStatus(Constants.MeetGroupConst.GroupStatus.REJECT.value());
        meetGroup.setInviteUserReadMark(Constants.NO);
        meetGroup.setAcceptUserReadMark(Constants.YES);
        meetGroup.setRejectMeetReason(reason);
        meetGroupService.updateById(meetGroup);

        eventPark.post(new MeetGroupRejectedEvent(this, meetGroup, acceptUserId));


        PushTarget pushTarget =new PushTarget();
        pushTarget.setUserId(meetGroup.getInviteUserId());
        pushTarget.addExtend("nickName",meetGroup.getAcceptUserBasicInfo().getNickname());
        pusher.doImmediatePush(PushScene.REJECTED_MEET_REQUEST, pushTarget);
        return meetGroup;
    }

    @Transactional(rollbackFor = Exception.class)
    public MeetGroupDao whistleMeetInvite(Long acceptUserId, Long meetGroupId, String reason) {
        MeetGroupDao meetGroup = checkGroupAcceptIsMe(meetGroupId, acceptUserId);
        if (meetGroup.getGroupStatus() != Constants.MeetGroupConst.GroupStatus.WAIT_ACCEPT.value()) {
            throw new IllegalStateException("见面群状态无效");
        }
        if (meetGroup.isExpiredInInviteTimeOut()) {
            throw new IllegalStateException("见面群邀请已过期");
        }
        this.refundTicket(meetGroupId, meetGroup.getInviteUserId(), TicketReturnTypeEnum.REFUSE);
        meetGroup.setMeetInviteUseTicketCount(0);
        meetGroup.setGroupStatus(Constants.MeetGroupConst.GroupStatus.WHISTLE.value());
        meetGroup.setInviteUserReadMark(Constants.NO);
        meetGroup.setAcceptUserReadMark(Constants.YES);
        meetGroup.setRejectMeetReason(reason);
        meetGroupService.updateById(meetGroup);

        eventPark.post(new MeetGroupWhistledEvent(this, meetGroup, acceptUserId));
        return meetGroup;
    }

    /**
     * 处理邀约超时的见面
     *
     * @param meetGroup 见面群
     * @throws IllegalStateException 不允许处理
     */
    @Transactional(rollbackFor = Exception.class)
    public void handleInviteTimeout(MeetGroupDao meetGroup) throws IllegalStateException {
        if (!meetGroup.isExpiredInInviteTimeOut()) {
            log.warn("见面群邀约未超时: {}", meetGroup);
            return;
        }
        if (meetGroup.getGroupStatus() != Constants.MeetGroupConst.GroupStatus.WAIT_ACCEPT.value()) {
            log.warn("见面群邀约状态无效: {}", meetGroup);
            return;
        }
        Long meetGroupId = meetGroup.getId();
        this.refundTicket(meetGroupId, meetGroup.getInviteUserId(), TicketReturnTypeEnum.TIME_OUT);

        meetGroup.setMeetInviteUseTicketCount(0);
        meetGroup.setGroupStatus(Constants.MeetGroupConst.GroupStatus.TIMEOUT.value());
        meetGroup.setInviteUserReadMark(Constants.NO);
        meetGroup.setAcceptUserReadMark(Constants.NO);

        meetGroupService.updateById(meetGroup);

        eventPark.post(new MeetGroupTimeoutEvent(this, meetGroup, null));

        log.info("见面群邀约已超时: {}", meetGroup.getId());
    }

    /**
     * 发送见面群消息
     * @param meetGroupMessageSendDto 发送消息的参数
     * @param userId 发送消息的用户ID
     * @return 发送的消息
     */
    @Transactional(rollbackFor = Exception.class)
    public MeetGroupMessage sendMessage(MeetGroupMessageSendDto meetGroupMessageSendDto, Long userId) {
        Long meetGroupId = meetGroupMessageSendDto.getMeetGroupId();
        MeetGroupDao meetGroupDao = checkGroupInviteOrAcceptIsMine(meetGroupId, userId);
        MeetGroupMessage message = new MeetGroupMessage()
                .setMeetGroupId(meetGroupId)
                .setRole(Constants.MeetGroupMessageConst.ROLE_USER)
                .setUserId(userId)
                .setContentType(meetGroupMessageSendDto.getContentType())
                .setContent(meetGroupMessageSendDto.getContent());
        meetGroupMessageService.save(message);

        // 对方是谁
        Long f2f = meetGroupDao.getTA(userId).getUserId();

        eventPark.post(new MeetGroupNewMessageEvent(this, message, f2f));

        PushTarget pushTarget =new PushTarget();
        pushTarget.setUserId(f2f);
        pushTarget.addExtend("nickName",meetGroupDao.getME(userId).getNickname());
        pusher.doImmediatePush(PushScene.NEW_NOTICE, pushTarget);
        return message;
    }

    /**
     * 发送见面群消息，来自AI。群内的AB成员都会收到消息。
     *
     * @param meetGroupMessageSendDto 发送消息的参数
     * @return 发送的消息
     */
    @Transactional(rollbackFor = Exception.class)
    public MeetGroupMessage sendMessageFromAssistant(MeetGroupMessageSendDto meetGroupMessageSendDto) {
        Long meetGroupId = meetGroupMessageSendDto.getMeetGroupId();
        MeetGroupDao meetGroupDao = meetGroupService.getDaoById(meetGroupId);
        if (meetGroupDao == null) {
            throw new IllegalArgumentException("找不到该见面群");
        }
        MeetGroupMessage message = new MeetGroupMessage()
                .setMeetGroupId(meetGroupId)
                .setRole(Constants.MeetGroupMessageConst.ROLE_ASSISTANT)
                .setContentType(meetGroupMessageSendDto.getContentType())
                .setContent(meetGroupMessageSendDto.getContent());
        meetGroupMessageService.save(message);

        eventPark.post(new MeetGroupNewMessageEvent(this, message, meetGroupDao.getInviteUserId()));
        eventPark.post(new MeetGroupNewMessageEvent(this, message, meetGroupDao.getAcceptUserId()));

        PushTarget pushTarget =new PushTarget();
        pushTarget.setUserId(meetGroupDao.getInviteUserId());
        pushTarget.addExtend("nickName",Constants.AI_NICK_NAME);
        pusher.doImmediatePush(PushScene.NEW_NOTICE, pushTarget);

        PushTarget acceptPushTarget =new PushTarget();
        acceptPushTarget.setUserId(meetGroupDao.getAcceptUserId());
        acceptPushTarget.addExtend("nickName",Constants.AI_NICK_NAME);
        pusher.doImmediatePush(PushScene.NEW_NOTICE, acceptPushTarget);
        return message;
    }

    /**
     * 选择见面地址
     * @param userId 用户ID
     * @param meetGroupAddressSelectDto 选择见面地址的参数
     */
    @Transactional(rollbackFor = Exception.class)
    public void selectMeetAddress(Long userId, MeetGroupAddressSelectDto meetGroupAddressSelectDto) {
        MeetGroupDao meetGroupDao = checkGroupInviteOrAcceptIsMine(meetGroupAddressSelectDto.getMeetGroupId(), userId);
        Integer meetStatus = meetGroupDao.getMeetStatus();
        if (!Objects.equals(meetStatus, Constants.MeetGroupConst.MeetAddressStatus.WAIT_SELECT.value())) {
            throw new IllegalStateException("当前状态不能选择见面地址");
        }
        if (!Objects.equals(meetGroupDao.getMeetSelectAddressUserId(), userId)) {
            throw new IllegalStateException("你暂时不能选择见面地址");
        }
        List<String> meetExpectTimesList = meetGroupAddressSelectDto.getMeetExpectTimesList();
        if (meetExpectTimesList.size() > 24) {
            throw new IllegalArgumentException("选择的时间不能超过24个");
        }
        meetGroupDao.setMeetAddress(meetGroupAddressSelectDto.getMeetAddress());
        meetGroupDao.setMeetExceptTimes(meetGroupAddressSelectDto.getMeetExceptTimes());
        meetGroupDao.setMeetStatus(Constants.MeetGroupConst.MeetAddressStatus.WAIT_CONFIRM.value());
        meetGroupService.updateById(meetGroupDao);

        eventPark.post(new MeetGroupMeetAddrEvent(this, meetGroupDao, userId));
    }

    /**
     * 确认见面时间
     * @param userId 用户ID
     * @param meetGroupAddressConfirmDto 同意见面的参数
     */
    @Transactional(rollbackFor = Exception.class)
    public void confirmMeetAddress(Long userId, MeetGroupAddressConfirmDto meetGroupAddressConfirmDto) {
        MeetGroupDao meetGroupDao = checkGroupInviteOrAcceptIsMine(meetGroupAddressConfirmDto.getMeetGroupId(), userId);
        if (!Objects.equals(meetGroupDao.getMeetStatus(), Constants.MeetGroupConst.MeetAddressStatus.WAIT_CONFIRM.value())) {
            throw new IllegalStateException("当前状态不能确认见面时间");
        }
        String meetTime = meetGroupAddressConfirmDto.getMeetTime();
        if (!meetGroupDao.getMeetExceptTimes().contains(meetTime)) {
            throw new IllegalArgumentException("时间段无效");
        }
        // 把字段值解析成 start time, end time
        // 2025-11-12 09:00-10:00
        String date = meetTime.split(" ")[0];
        String times = meetTime.split(" ")[1];
        String s = times.split("-")[0];
        String e = times.split("-")[1];

        LocalDateTime start = LocalDateTime.parse(date + " " + s, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"));
        LocalDateTime end = LocalDateTime.parse(date + " " + e, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"));

        meetGroupDao.setMeetTime(meetTime);
        meetGroupDao.setMeetTimeStart(start);
        meetGroupDao.setMeetTimeEnd(end);
        meetGroupDao.setMeetStatus(Constants.MeetGroupConst.MeetAddressStatus.CONFIRM.value());
        meetGroupService.updateById(meetGroupDao);

        // 发送同意消息
        UserBasicInfo me = meetGroupDao.getME(userId);

        String content = String.format("%s已经确认见面地点和时间，见面地点：%s，见面时间：%s",
                me.getNickname(), meetGroupDao.getMeetAddress(), meetGroupDao.getMeetTime());
        MeetGroupMessageSendDto meetGroupMessageSendDto = new MeetGroupMessageSendDto()
                .setContentType(Constants.MeetGroupMessageConst.CONTENT_TYPE_TEXT)
                .setContent(content)
                .setMeetGroupId(meetGroupAddressConfirmDto.getMeetGroupId());
        this.sendMessageFromAssistant(meetGroupMessageSendDto);

        eventPark.post(new MeetGroupMeetAddrEvent(this, meetGroupDao, userId));
    }

    /**
     * 拒绝见面地址
     * @param meetGroupAddressRejectDto 拒绝见面地址的参数
     */
    @Transactional(rollbackFor = Exception.class)
    public void rejectMeetAddress(Long userId, MeetGroupAddressRejectDto meetGroupAddressRejectDto) {
        MeetGroupDao meetGroupDao = checkGroupInviteOrAcceptIsMine(meetGroupAddressRejectDto.getMeetGroupId(), userId);
        if (!Objects.equals(meetGroupDao.getMeetStatus(), Constants.MeetGroupConst.MeetAddressStatus.WAIT_CONFIRM.value())) {
            throw new IllegalStateException("当前状态不能拒绝见面地址");
        }
        meetGroupDao.setRejectMeetReason(meetGroupAddressRejectDto.getReason());
        meetGroupDao.setMeetStatus(Constants.MeetGroupConst.MeetAddressStatus.WAIT_SELECT.value());
        meetGroupService.updateById(meetGroupDao);

        // 发送拒绝消息
        UserBasicInfo me = meetGroupDao.getME(userId);
        UserBasicInfo ta = meetGroupDao.getTA(userId);

        String content = String.format("@%s，由于%s的时间不合适，麻烦你重新选择一下见面地址和时间哦。", ta.getNickname(), me.getNickname());
        MeetGroupMessageSendDto meetGroupMessageSendDto = new MeetGroupMessageSendDto()
                .setContentType(Constants.MeetGroupMessageConst.CONTENT_TYPE_TEXT)
                .setContent(content)
                .setMeetGroupId(meetGroupAddressRejectDto.getMeetGroupId());
        this.sendMessageFromAssistant(meetGroupMessageSendDto);

        eventPark.post(new MeetGroupMeetAddrEvent(this, meetGroupDao, userId));
    }

    /**
     * 确认见面结果
     * @param userId 用户ID
     * @param meetGroupConfirmMeetResultDto 见面结果
     */
    public void confirmMeetResult(Long userId, MeetGroupConfirmMeetResultDto meetGroupConfirmMeetResultDto) {
        Long meetGroupId = meetGroupConfirmMeetResultDto.getMeetGroupId();
        MeetGroupDao meetGroupDao = checkGroupInviteOrAcceptIsMine(meetGroupId, userId);
        if (!Objects.equals(meetGroupDao.getMeetStatus(), Constants.MeetGroupConst.MeetAddressStatus.CONFIRM.value())) {
            throw new IllegalStateException("仅确认了见面地点和时间的才能确认见面结果");
        }
        LocalDateTime meetTimeStart = meetGroupDao.getMeetTimeStart();
        if (null == meetTimeStart || meetTimeStart.isAfter(LocalDateTime.now())) {
            throw new IllegalStateException("见面时间未开始");
        }
        Integer meetResult = meetGroupConfirmMeetResultDto.getMeetResult();
        String description = meetGroupConfirmMeetResultDto.getDescription();
        meetGroupDao.updateMeetResultForMe(userId, meetResult, description);

        boolean inviteNotMeet = Objects.equals(meetGroupDao.getMeetInviteUserResult(), Constants.MeetGroupConst.MeetResult.NOT_MEET.value());
        boolean acceptNotMeet = Objects.equals(meetGroupDao.getMeetAcceptUserResult(), Constants.MeetGroupConst.MeetResult.NOT_MEET.value());
        // 有一方提交未见面，则结束，表示未见面。

        if (inviteNotMeet || acceptNotMeet) {
            // 重新设置未见面流程，将见面地址等信息清空，重新选择。
            meetGroupDao.setMeetStatus(Constants.MeetGroupConst.MeetAddressStatus.WAIT_SELECT.value());
            meetGroupDao.setMeetAddress(null);
            meetGroupDao.setMeetExceptTimes(null);
            meetGroupDao.setMeetTime(null);
            meetGroupDao.setMeetTimeStart(null);
            meetGroupDao.setMeetTimeEnd(null);
        }
        // 如果两方都确认已见面
        boolean inviteMeet = Objects.equals(meetGroupDao.getMeetInviteUserResult(), Constants.MeetGroupConst.MeetResult.MEET.value());
        boolean acceptMeet = Objects.equals(meetGroupDao.getMeetAcceptUserResult(), Constants.MeetGroupConst.MeetResult.MEET.value());
        if (inviteMeet && acceptMeet) {
            meetGroupDao.setGroupStatus(Constants.MeetGroupConst.GroupStatus.MEET_SUCCESS.value());
        }

        meetGroupService.updateById(meetGroupDao);

        eventPark.post(new MeetGroupMeetAddrEvent(this, meetGroupDao, userId));
    }

    /**
     * 获取虚拟电话号码
     *
     * @param meetGroupId 见面群ID
     * @param userId 用户ID
     * @return 虚拟号码
     */
    public String getVirtualPhoneNumber(Long meetGroupId, Long userId) {
        MeetGroupDao meetGroupDao = checkGroupInviteOrAcceptIsMine(meetGroupId, userId);

        Long inviteUserId = meetGroupDao.getInviteUserId();
        Long acceptUserId = meetGroupDao.getAcceptUserId();

        User inviteUser = userService.getById(inviteUserId);
        User acceptUser = userService.getById(acceptUserId);

        boolean iAmInvite = Objects.equals(inviteUserId, userId);

        String a = iAmInvite ? inviteUser.getMobile() : acceptUser.getMobile();
        String b = iAmInvite ? acceptUser.getMobile() : inviteUser.getMobile();

        VirtualPhoneNumberBindResult bindResult = virtualPhoneNumberService.rebind(a, b);
        return bindResult.getVirtualNumber();
    }

    /**
     * 检查该见面群是否属于该用户
     * @param meetGroupId 见面群ID
     * @param userId 用户ID
     * @throws IllegalArgumentException 如果该见面群不属于该用户，则抛出异常
     */
    public MeetGroupDao checkGroupInviteOrAcceptIsMine(Long meetGroupId, Long userId) throws IllegalArgumentException {
        MeetGroupDao meetGroup = meetGroupService.getDaoById(meetGroupId);
        boolean is = null != meetGroup
                && (Objects.equals(meetGroup.getInviteUserId(), userId) || Objects.equals(meetGroup.getAcceptUserId(), userId));
        if (!is) {
            throw new IllegalArgumentException("无效的见面群");
        }
        return meetGroup;
    }

    /**
     * 检查该见面群的受邀者是否属于该用户
     * @param meetGroupId 见面群ID
     * @param userId 用户ID
     * @throws IllegalArgumentException 如果该见面群的受邀者不是该用户，则抛出异常
     */
    public MeetGroupDao checkGroupAcceptIsMe(Long meetGroupId, Long userId) {
        MeetGroupDao meetGroup = meetGroupService.getDaoById(meetGroupId);
        boolean is = null != meetGroup
                && Objects.equals(meetGroup.getAcceptUserId(), userId);
        if (!is) {
            throw new IllegalArgumentException("无效的见面群");
        }
        return meetGroup;
    }

    /**
     * 退回相亲券，如果没有需要退回，则不处理
     *
     * @param meetGroupId 见面群ID
     * @param userId 用户ID
     */
    private void refundTicket(Long meetGroupId, Long userId, TicketReturnTypeEnum returnType) {
        QueryWrapper<UserMeetTicketRecord> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(UserMeetTicketRecord.CONNECT_ID, meetGroupId)
                .eq(UserMeetTicketRecord.TYPE, TicketRecordTypeEnum.USE.getOrdinal())
                .eq(UserMeetTicket.USER_ID, userId);
        UserMeetTicketRecord userMeetTicketRecord = userMeetTicketRecordService.getOne(queryWrapper);
        if (userMeetTicketRecord == null) {
            log.info("见面群：{}，用户：{}，没有使用红豆，不需要退回", meetGroupId, userId);
            return;
        }
        // 然后调用退回接口

        ReturnTicketOperateDto ticketOperateDto = new ReturnTicketOperateDto()
                .setAmount(userMeetTicketRecord.getAmount())
                .setConnectId(meetGroupId)
                .setUserId(userId)
                .setReturnType(returnType);
        userMeetTicketService.returnTicket(ticketOperateDto);
        log.info("见面群：{}，用户：{}，退豆成功！退豆情况：{}", meetGroupId, userId, ticketOperateDto);
    }

    /**
     * 过滤出第一个女性用户
     * @param userBasicInfoMap 用户基本信息
     * @return 女性用户
     */
    private UserBasicInfo filterFemale(Map<Long, UserBasicInfo> userBasicInfoMap) {
        return userBasicInfoMap.values().stream()
                .filter(e -> e.getGender() == Constants.UserGender.FEMALE)
                .findFirst()
                .orElse(null);
    }

    /**
     * 获取第一个男性用户
     * @param userBasicInfoMap 用户基本信息
     * @return 男性用户
     */
    private UserBasicInfo filterMale(Map<Long, UserBasicInfo> userBasicInfoMap) {
        return userBasicInfoMap.values().stream()
               .filter(e -> e.getGender() == Constants.UserGender.MALE)
               .findFirst()
               .orElse(null);
    }

    /**
     * 获取第一个女性用户
     * @param userId 用户ID
     * @return female
     */
    private UserBasicInfo getFirstFemale(Object...userId) {
        Map<Long, UserBasicInfo> userBasicInfoMap = getUserBasicInfoMap(userId);
        return filterFemale(userBasicInfoMap);
    }

    /**
     * 获取用户基本信息
     * @param userId 用户ID
     * @return 用户基本信息
     */
    private Map<Long, UserBasicInfo> getUserBasicInfoMap(Object...userId) {
        QueryWrapper<UserBasicInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .in(UserBasicInfo::getUserId, userId);
        List<UserBasicInfo> list = userBasicInfoService.list(queryWrapper);
        return list.stream().collect(Collectors.toMap(UserBasicInfo::getUserId, Function.identity()));
    }
}
