package com.bxm.lovelink.common.dal.manager;

import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.dao.MeetGroupDao;
import com.bxm.lovelink.common.dal.service.IMeetGroupService;
import com.bxm.lovelink.common.dal.service.IUserService;
import com.bxm.warcar.utils.TypeHelper;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 用来处理已经失效的未读消息数，比如用户注销、见面群删除后，未读消息数量需要处理掉。
 * 否则用户的未读消息数会一直存在，永远无法清掉。
 *
 * @author Allen Hu
 * @date 2025/7/7
 */
@Service
public class MeetGroupMessageInvalidUnreadCountManager implements InitializingBean, DisposableBean {

    private final IMeetGroupService meetGroupService;
    private final IUserService userService;
    private final ExecutorService executor = Executors.newSingleThreadExecutor();
    private final BlockingQueue<QueueItem> queue = new LinkedBlockingQueue<>(1000);

    public MeetGroupMessageInvalidUnreadCountManager(IMeetGroupService meetGroupService, IUserService userService) {
        this.meetGroupService = meetGroupService;
        this.userService = userService;
    }

    public void add2Queue(QueueItem queueItem) {
        if (Objects.nonNull(queueItem)) {
            queue.offer(queueItem);
        }
    }

    @Override
    public void afterPropertiesSet() {
        this.executor.execute(() -> {
            for (;;) {
                try {
                    QueueItem queueItem = queue.take();
                    this.removeIf(queueItem);
                } catch (InterruptedException ignored) {
                }
            }
        });
    }

    @Override
    public void destroy() {
        this.executor.shutdownNow();
    }

    private void removeIf(QueueItem queueItem) {
        Long userId = queueItem.getUserId();
        Map<String, Integer> meetGroupId2Count = queueItem.getMeetGroupId2Count();
        // 检查群是否还存在
        List<WaitRemoveItem> waitRemoveItems = meetGroupId2Count.keySet()
                .stream().filter(meetGroupId -> {
                    MeetGroupDao meetGroup = meetGroupService.getDaoById(TypeHelper.castToLong(meetGroupId));
                    if (Objects.isNull(meetGroup) || Objects.equals(meetGroup.getDeleted(), Constants.DELETED)) {
                        // need remove
                        return true;
                    }
                    // 判断对方是否已经注销？
                    UserBasicInfo ta = meetGroup.getTA(userId);
                    Long taUserId = ta.getUserId();
                    User taUser = userService.getById(taUserId);
                    if (Objects.isNull(taUser) || Objects.equals(taUser.getDeleted(), Constants.DELETED)) {
                        // need remove;
                        return true;
                    }
                    return false;
                })
                .map(new Function<String, WaitRemoveItem>() {
                    @Override
                    public WaitRemoveItem apply(String meetGroupId) {
                        return new WaitRemoveItem(userId, Long.valueOf(meetGroupId));
                    }
                })
                .collect(Collectors.toList());
        Consumer<List<WaitRemoveItem>> removeHandler = queueItem.getRemoveHandler();
        if (removeHandler != null) {
            removeHandler.accept(waitRemoveItems);
        }
    }

    @Data
    @AllArgsConstructor
    public static class QueueItem {
        private Long userId;
        private Map<String, Integer> meetGroupId2Count;
        private Consumer<List<WaitRemoveItem>> removeHandler;
    }

    @Data
    @AllArgsConstructor
    public static class WaitRemoveItem {
        private Long userId;
        private Long meetGroupId;
    }
}
