package com.bxm.lovelink.common.dal.manager;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bxm.lovelink.common.chat.*;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.ChatSession;
import com.bxm.lovelink.common.dal.entity.ChatSessionMessage;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.entity.dto.chat.WecomChatDto;
import com.bxm.lovelink.common.dal.manager.parameter.ParameterBuilderFactory;
import com.bxm.lovelink.common.dal.manager.parameter.ParameterBuilderRequest;
import com.bxm.lovelink.common.dal.service.IChatSessionMessageService;
import com.bxm.lovelink.common.dal.service.IChatSessionService;
import com.bxm.lovelink.common.dal.service.IUserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.connector.ClientAbortException;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiConsumer;

/**
 * @author tangxiao
 * @date 2025/7/15
 */
@Slf4j
@Component
public class WecomChatManager {

    private final IUserService userService;
    private final IChatSessionService chatSessionService;
    private final IChatSessionMessageService chatSessionMessageService;
    private final ChatService chatService;
    private final ParameterBuilderFactory parameterBuilderFactory;

    public WecomChatManager(IUserService userService, IChatSessionService chatSessionService, IChatSessionMessageService chatSessionMessageService, ChatService chatService, ParameterBuilderFactory parameterBuilderFactory) {
        this.userService = userService;
        this.chatSessionService = chatSessionService;
        this.chatSessionMessageService = chatSessionMessageService;
        this.chatService = chatService;
        this.parameterBuilderFactory = parameterBuilderFactory;
    }

    public void streamChat(User user, WecomChatDto wecomChatDto,
                           ByteWriter<byte[]> writeConsumer,
                           SimpleHandler flusher,
                           CompleteConsumer completeConsumer,
                           BiConsumer<ClientAbortException, String> clientAbortExceptionStringBiConsumer) {
        Long userId = user.getId();

        LambdaQueryWrapper<ChatSession> queryWrapper = Wrappers.lambdaQuery(ChatSession.class)
                .eq(ChatSession::getUserId, userId)
                .eq(ChatSession::getSceneType, wecomChatDto.getSceneType());
        ChatSession session = chatSessionService.getOne(queryWrapper);
        boolean isFistChat = null == session;
        if (isFistChat) {
            session = new ChatSession();
            session.setUserId(userId);
            session.setSceneType(Constants.Chat.SCENE_TYPE_WECOM);
            session.setExt(wecomChatDto.getUnionId());
            this.chatSessionService.save(session);
        }

        // 如果用户首次聊天 要带上欢迎语
        List<ChatSessionMessage> messageList = new ArrayList<>();
        if (isFistChat && wecomChatDto.getWelcomeContent() != null) {
            messageList.add(wecomChatDto.to()
                    .setContent(wecomChatDto.getWelcomeContent())
                    .setSessionId(session.getId())
                    .setRole(Constants.Chat.ROLE_ASSISTANT)
            );
        }

        // 保存用户的消息
        messageList.add(wecomChatDto.to()
                .setSessionId(session.getId())
                .setRole(Constants.Chat.ROLE_USER));

        messageList.forEach(chatSessionMessageService::save);

        long start = System.currentTimeMillis();

        ChatRequest chatRequest = ChatRequest.builder()
                .user(user)
                .chatSession(session)
                .chatSessionMessages(messageList)
                .parameter(parameterBuilderFactory.build(
                        ParameterBuilderRequest.builder()
                                .userId(userId)
                                .chatSession(session)
                                .apiChatDto(wecomChatDto)
                                .me(userService.getCompositeById(userId))
                                .build()
                ))
                .writeConsumer(writeConsumer)
                .flusher(flusher)
                .complete(messages -> {
                    ChatSessionMessage lastMessage = messages.get(messages.size() - 1);
                    chatSessionMessageService.save(lastMessage);
                    if (null != completeConsumer) {
                        completeConsumer.accept(messages);
                    }
                    log.info("企微聊天大模型耗时{}ms", System.currentTimeMillis() - start);
                })
                .clientAbortExceptionStringBiConsumer(clientAbortExceptionStringBiConsumer)
                .build();
        chatService.stream(chatRequest);
    }
}
