package com.bxm.lovelink.common.dal.manager.parameter.builders;

import cn.hutool.core.io.IoUtil;
import com.bxm.lovelink.common.dal.entity.Region;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserComposite;
import com.bxm.lovelink.common.dal.entity.UserTag;
import com.bxm.lovelink.common.dal.manager.parameter.ParameterBuilder;
import com.bxm.lovelink.common.dal.manager.parameter.ParameterBuilderRequest;
import com.bxm.lovelink.common.dal.manager.parameter.parser.PromptParser;
import com.bxm.lovelink.common.dal.manager.parameter.parser.VelocityPromptParser;
import com.bxm.lovelink.common.dal.service.IRegionService;
import com.bxm.lovelink.common.dal.service.IUserTagService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Allen Hu
 * @date 2025/7/1
 */
@Slf4j
@Service
public class UserProfileBuilder implements ParameterBuilder {

    private final IRegionService regionService;
    private final IUserTagService userTagService;
    private final Resource resource;

    public UserProfileBuilder(IRegionService regionService, IUserTagService userTagService) {
        this.regionService = regionService;
        this.userTagService = userTagService;
        this.resource = new ClassPathResource("llm_param_for_user_profile.vm");
    }

    @Override
    public String key() {
        return "user_a_profile";
    }

    protected UserComposite getUserComposite(ParameterBuilderRequest parameterBuilderRequest) {
        return parameterBuilderRequest.getMe();
    }

    @Override
    public Object apply(ParameterBuilderRequest parameterBuilderRequest) {
        UserComposite userComposite = getUserComposite(parameterBuilderRequest);
        if (userComposite == null) {
            return null;
        }
        Map<String, Object> variables = new HashMap<>();
        variables.put("userComposite", userComposite);
        UserBasicInfo userBasicInfo = userComposite.getUserBasicInfo();
        if (null != userBasicInfo) {
            Map<Integer, String> regionMap = regionService.fetchAll(Region::getFullName);
            variables.put("currentPlaceDistrict", regionMap.get(userBasicInfo.getCurrentPlaceDistrict()));
            variables.put("nativePlaceDistrict", regionMap.get(userBasicInfo.getNativePlaceDistrict()));
            variables.put("interestList", fetchInterestTags(userBasicInfo.getInterestTag()));
        }
        PromptParser parser = new VelocityPromptParser(variables);
        String string = readFull(resource);
        return parser.process(string);
    }

    protected String readFull(Resource resource) {
        try {
            return IoUtil.read(resource.getInputStream(), StandardCharsets.UTF_8);
        } catch (IOException e) {
            log.error("【velocity-template】读取模板异常，resource:{}", resource);
            return null;
        }
    }

    protected List<String> fetchInterestTags(List<Long> interestTagIds) {
        if (CollectionUtils.isEmpty(interestTagIds)) {
            return Collections.emptyList();
        }
        Map<Long, String> userTagMap = userTagService.fetchAll(UserTag::getTagName);
        return interestTagIds.stream()
                .filter(Objects::nonNull)
                .map(userTagMap::get)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

}
