package com.bxm.lovelink.common.dal.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.ActivityRegistrationData;
import com.bxm.lovelink.common.dal.entity.vo.activity.ActivityPayConfigVo;
import com.bxm.lovelink.common.dal.service.IActivityUserMutualSelectionService;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserOrder;
import com.bxm.lovelink.common.dal.entity.dto.activity.evaluation.ActivityRegisterDto;
import com.bxm.lovelink.common.dal.entity.vo.activity.registration.ActivityRegistrationResultVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.usermutualselection.ActivityUserVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.mapper.ActivityRegistrationMapper;
import com.bxm.lovelink.common.dal.service.IActivityRegistrationService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.dal.service.IUserOrderService;
import com.bxm.warcar.id.IdGenerator;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 活动报名表 服务实现类
 * </p>
 *
 * @author dll
 * @since 2025-06-24
 */
@Slf4j
@Service
public class ActivityRegistrationServiceImpl extends ServiceImpl<ActivityRegistrationMapper, ActivityRegistration> implements IActivityRegistrationService {
    private final ActivityMapper activityMapper;
    private final IdGenerator idGenerator;
    private final IUserOrderService userOrderService;
    private final RedissonClient redissonClient;
    private final IActivityUserMutualSelectionService activityUserMutualSelectionService;
    private final Pair pair;

    public ActivityRegistrationServiceImpl(ActivityMapper activityMapper, @Qualifier("orderIdGenerator") IdGenerator idGenerator, IUserOrderService userOrderService, RedissonClient redissonClient, IActivityUserMutualSelectionService activityUserMutualSelectionService, Pair pair) {
        this.activityMapper = activityMapper;
        this.idGenerator = idGenerator;
        this.userOrderService = userOrderService;
        this.redissonClient = redissonClient;
        this.activityUserMutualSelectionService = activityUserMutualSelectionService;
        this.pair = pair;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public ActivityRegistrationResultVo register(ActivityRegisterDto dto, UserBasicInfo userBasicInfo) {
        Long userId = userBasicInfo.getUserId();
        Long activityId = dto.getActivityId();
        LocalDateTime now = LocalDateTime.now();
        Activity activity = activityMapper.selectById(activityId);
        validateActivity(activity);
        RLock lock = redissonClient.getLock(RedisKeys.activityRegisterLock(userId, activityId).generateKey());
        boolean locked = lock.tryLock();
        if (!locked) {
            throw new IllegalStateException("正在报名中，请稍后再试");
        }
        try {
            // 判断是否已经报名
            ActivityRegistration activityRegistration = getActivityRegistration(activityId, userId, Constants.ActivityRegistrationStatus.SUCCESS);
            if (activityRegistration != null) {
                throw new IllegalStateException("已经报名过了");
            }
            // 查询是否有待支付订单
            ActivityRegistration waitPay = getWaitPayActivityRegistration(activityId, userId);
            if (waitPay != null) {
                UserOrder userOrder = userOrderService.query(waitPay.getOrderNo());
                if (Objects.nonNull(userOrder) && userOrder.getStatus() < Constants.UserOrder.STATUS_SUCCESS) {
                    return ActivityRegistrationResultVo.buildWaitPayResult(activityId, waitPay.getOrderNo(), userOrder.getTotalPrice(), getAppleProductId(userOrder.getPaymentAmount()));
                }
            }
            // 校验报名人数是否已满
            Integer canRegister = activityMapper.canRegister(activityId, userBasicInfo.getGender());
            if (Objects.isNull(canRegister) || !canRegister.equals(Constants.YES)) {
                return ActivityRegistrationResultVo.buildFailResult(activityId, Constants.RegistrationResult.FULL);
            }
            // 免费报名
            if (isFree(activity, userId)) {
                boolean result = registerUser(dto.getActivityId(), userBasicInfo, Constants.ActivityRegistrationStatus.SUCCESS, false, null);
                if (!result) {
                    return ActivityRegistrationResultVo.buildFailResult(activityId, Constants.RegistrationResult.FULL);
                }
                return ActivityRegistrationResultVo.buildFreeResult(activityId, BigDecimal.ZERO);
            }
            // 使用赠送报名机会
            if (useFreeQualification(userId, activity.getActivityTypeId())) {
                boolean result = registerUser(dto.getActivityId(), userBasicInfo, Constants.ActivityRegistrationStatus.SUCCESS, false, null);
                if (!result) {
                    return ActivityRegistrationResultVo.buildFailResult(activityId, Constants.RegistrationResult.FULL);
                }
                return ActivityRegistrationResultVo.buildFreeResult(activityId, BigDecimal.ZERO);
            }
            // 早鸟票
            if (!dto.getIsOriginalPriceRegister() && hasDiscount(activity)) {
                if (hasValidDiscount(activity, now)) {
                    String orderNo = createOrder(dto, userId, activity, activity.getDiscountPrice());
                    boolean result = registerUser(dto.getActivityId(), userBasicInfo, Constants.ActivityRegistrationStatus.WAIT_PAY, true, orderNo);
                    if (!result) {
                        // 回滚早鸟票名额
                        releaseDiscountRegisterCount(activity);
                        return ActivityRegistrationResultVo.buildFailResult(activityId, Constants.RegistrationResult.FULL);
                    }
                    return ActivityRegistrationResultVo.buildPayResult(activityId, orderNo, activity.getDiscountPrice(), getAppleProductId(activity.getDiscountPrice()));
                }
                return ActivityRegistrationResultVo.buildFailResult(activityId, Constants.RegistrationResult.DISCOUNT_SOLD_OUT, activity.getRegisterPrice());
            }
            // 正常付费报名
            String orderNo = createOrder(dto, userId, activity, activity.getRegisterPrice());
            // 新增报名记录
            boolean result = registerUser(dto.getActivityId(), userBasicInfo, Constants.ActivityRegistrationStatus.WAIT_PAY, false, orderNo);
            if (!result) {
                return ActivityRegistrationResultVo.buildFailResult(activityId, Constants.RegistrationResult.FULL);
            }
            return ActivityRegistrationResultVo.buildPayResult(activityId, orderNo, activity.getRegisterPrice(), getAppleProductId(activity.getRegisterPrice()));
        } finally {
            lock.unlock();
        }
    }

    private void releaseDiscountRegisterCount(Activity activity) {
        if (Objects.nonNull(activity.getDiscountLimitType()) && activity.getDiscountLimitType() == Constants.DiscountLimitType.COUNT){
            activityMapper.releaseDiscountRegisterCount(activity.getId());
        }
    }

    private boolean registerUser(Long activityId, UserBasicInfo userBasicInfo, int status, boolean discount, String orderNo) {
        // 修改名额
        boolean success = tryRegister(activityId, userBasicInfo);
        if (!success) {
            return false;
        }
        // 插入报名记录
        Integer insert = insertRegistration(activityId, userBasicInfo, status, discount, orderNo);
        return insert == 1;
    }

    private boolean isFree(Activity activity, Long userId) {
        if (Objects.isNull(activity.getRegisterPrice()) || activity.getRegisterPrice().compareTo(BigDecimal.ZERO) <= 0) {
            return true;
        }
        if (Constants.YES == activity.getNewUserFree() && Objects.nonNull(userId)) {
            // 用户是否成功参加过活动
            return baseMapper.selectCount(new LambdaQueryWrapper<ActivityRegistration>()
                    .eq(ActivityRegistration::getUserId, userId)
                    .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS)) <= 0;
        }
        return false;
    }

    private boolean useFreeQualification(Long userId, Long typeId) {
        List<ActivityRegistration> list = baseMapper.selectList(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getUserId, userId)
                .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS)
                .eq(ActivityRegistration::getFreeActivityTypeId, typeId));
        if (CollectionUtils.isNotEmpty(list)) {
            ActivityRegistration freeReg = list.get(0);
            freeReg.setFreeActivityTypeId(null);
            baseMapper.updateById(freeReg);
            return true;
        }
        return false;
    }

    private boolean hasValidDiscount(Activity activity, LocalDateTime now) {
        int type = activity.getDiscountLimitType();
        if (type == Constants.DiscountLimitType.DATE) {
            if (activity.getDiscountStartTime() != null
                    && activity.getDiscountEndTime() != null
                    && !now.isBefore(activity.getDiscountStartTime())
                    && !now.isAfter(activity.getDiscountEndTime())) {
                return activityMapper.incrementDiscountRegisterCount(activity.getId()) == 1;
            }
        } else if (type == Constants.DiscountLimitType.COUNT) {
            return activityMapper.lockDiscountRegisterCount(activity.getId()) == 1;
        }
        return false;
    }

    private boolean hasDiscount(Activity activity) {
        return activity.getDiscountPrice() != null && activity.getDiscountLimitType() != null && activity.getDiscountLimitType() != Constants.NO;
    }

    private void validateActivity(Activity activity) {
        if (activity == null) {
            throw new IllegalStateException("活动不存在");
        }
        LocalDateTime now = LocalDateTime.now();
        if (activity.getRegisterStartTime().isAfter(now)) {
            throw new IllegalStateException("活动报名还未开始");
        }
        if (activity.getRegisterEndTime().isBefore(now)) {
            throw new IllegalStateException("活动报名已结束");
        }
        if (activity.getActivityEndTime().isBefore(now)) {
            throw new IllegalStateException("活动已结束");
        }
    }

    private boolean tryRegister(Long activityId, UserBasicInfo userBasicInfo) {
        boolean result = (userBasicInfo.getGender() == Constants.UserGender.MALE)
                ? activityMapper.tryUpdateMaleCount(activityId) == 1
                : activityMapper.tryUpdateFemaleCount(activityId) == 1;
        if (result) {
            log.info("报名成功，活动ID：{}，用户ID：{}, 性别：{}", activityId, userBasicInfo.getUserId(), userBasicInfo.getGender());
        }
        return result;
    }

    private ActivityRegistration getActivityRegistration(Long activityId, Long userId, Integer status) {
        return baseMapper.selectOne(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getActivityId, activityId)
                .eq(ActivityRegistration::getUserId, userId)
                .eq(ActivityRegistration::getStatus, status));
    }

    private ActivityRegistration getWaitPayActivityRegistration(Long activityId, Long userId) {
        return baseMapper.selectOne(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getActivityId, activityId)
                .eq(ActivityRegistration::getUserId, userId)
                .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.WAIT_PAY)
                .orderByDesc(ActivityRegistration::getId)
                .last("LIMIT 1"));
    }

    private Integer insertRegistration(Long activityId, UserBasicInfo user, int status, boolean discount, String orderNo) {
        ActivityRegistration registration = new ActivityRegistration();
        registration.setActivityId(activityId);
        registration.setUserId(user.getUserId());
        registration.setGender(user.getGender());
        registration.setStatus(status);
        registration.setDiscountApplied(discount ? Constants.YES : Constants.NO);
        registration.setRegistrationTime(LocalDateTime.now());
        registration.setOrderNo(orderNo);
        return baseMapper.insert(registration);
    }

    private String createOrder(ActivityRegisterDto dto, Long userId, Activity activity, BigDecimal price) {
        String orderNum = idGenerator.next();
        UserOrder order = new UserOrder()
                .setOrderNum(orderNum)
                .setUserId(userId)
                .setMeetTicketProductId(0L)
                .setChannel(dto.getChannel())
                .setProductType(Constants.MeetTicketProduct.ACTIVITY_PAY)
                .setProductName(activity.getName())
                .setProductDesc(activity.getIntro())
                .setTotalPrice(price)
                .setPaymentAmount(BigDecimal.ZERO)
                .setExtJson(String.valueOf(activity.getId()))
                .setStatus(Constants.UserOrder.STATUS_WAIT_FOR_PAY);

        if (!userOrderService.save(order)) {
            throw new IllegalStateException("报名失败");
        }
        return orderNum;
    }

    /**
     * 计算活动报名价格
     */
    @Override
    public BigDecimal calculateActivityPrice(Activity activity, Long userId) {
        boolean free = isFree(activity, userId);
        if (free) {
            return BigDecimal.ZERO;
        }
        if (Objects.nonNull(userId)) {
            // 查询是否有免费报名机会
            List<ActivityRegistration> registrations = baseMapper.selectList(new LambdaQueryWrapper<ActivityRegistration>()
                    .eq(ActivityRegistration::getUserId, userId)
                    .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS)
                    .eq(ActivityRegistration::getFreeActivityTypeId, activity.getActivityTypeId()));
            if (CollectionUtils.isNotEmpty(registrations)) {
                return BigDecimal.ZERO;
            }
        }
        if (Objects.nonNull(activity.getDiscountPrice()) && Objects.nonNull(activity.getDiscountLimitType()) && activity.isDiscountApplied()) {
            return activity.getDiscountPrice();
        }
        return activity.getRegisterPrice();
    }

    @Override
    public ActivityRegistration getOneByOrderNo(String orderNo) {
        return baseMapper.selectOne(new LambdaQueryWrapper<ActivityRegistration>().
                eq(ActivityRegistration::getOrderNo, orderNo));
    }

    @Override
    public Map<Long, List<ActivityUserVo>> getRegiactUserMap(List<Long> activityIdsList) {
        List<ActivityUserVo> result = baseMapper.regiactivityUserList(activityIdsList);
        return result.stream()
                .collect(Collectors.groupingBy(ActivityUserVo::getActivityId));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void userCancel(Long userId) {
        List<ActivityRegistration> registrations = baseMapper.selectList(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getUserId, userId));
        if (CollectionUtils.isEmpty(registrations)) {
            return;
        }
        Integer gender = CollUtil.getFirst(registrations).getGender();
        for (ActivityRegistration registration : registrations) {
            Activity activity = activityMapper.selectById(registration.getActivityId());
            if (Objects.isNull(activity) || !activity.getActivityStartTime().isAfter(LocalDateTime.now())) {
                continue;
            }
            // 处理未开始的活动
            if (registration.getStatus().equals(Constants.ActivityRegistrationStatus.SUCCESS) || registration.getStatus().equals(Constants.ActivityRegistrationStatus.WAIT_PAY)) {
                //  减少报名名额
                boolean isMale = gender == Constants.UserGender.MALE;
                if (isMale) {
                    activityMapper.releaseMaleRegisterCount(activity.getId());
                } else {
                    activityMapper.releaseFemaleRegisterCount(activity.getId());
                }
                if (Objects.equals(Constants.YES, registration.getDiscountApplied())) {
                    activityMapper.releaseDiscountRegisterCount(activity.getId());
                }
                if (registration.getStatus().equals(Constants.ActivityRegistrationStatus.WAIT_PAY) && StringUtils.isNotBlank(registration.getOrderNo())) {
                    userOrderService.update(new LambdaUpdateWrapper<UserOrder>()
                            .eq(UserOrder::getOrderNum, registration.getOrderNo())
                            .eq(UserOrder::getUserId, userId)
                            .set(UserOrder::getStatus, Constants.UserOrder.STATUS_CANCEL));
                }
            }
            // 删除报名记录
            registration.setDeleted(Constants.YES);
            baseMapper.updateById(registration);
        }
    }

    @Override
    public ActivityPayConfigVo getPayConfig() {
        String payConfig = pair.get(Constants.PairKey.ACTIVITY_PAY_CONFIG).of();
        if (StringUtils.isBlank(payConfig)) {
            return null;
        }
        return JsonHelper.convert(payConfig, ActivityPayConfigVo.class);
    }

    @Override
    public Map<Long, ActivityRegistrationData> getActivityRegistrationDataMap(List<Long> activityIdsList) {
        if (CollectionUtils.isEmpty(activityIdsList)) {
            return Collections.emptyMap();
        }
        return getActivityRegistrationDataMap(getRegiactUserMap(activityIdsList));
    }

    @Override
    public Map<Long, ActivityRegistrationData> getActivityRegistrationDataMap(Map<Long, List<ActivityUserVo>> activityUserMap) {
        return activityUserMap.entrySet().stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        entry -> {
                            List<ActivityUserVo> userList = entry.getValue();
                            long maleCount = userList.stream()
                                    .filter(user -> user.getGender() != null && user.getGender() == 1)
                                    .count();
                            long femaleCount = userList.stream()
                                    .filter(user -> user.getGender() != null && user.getGender() == 0)
                                    .count();
                            ActivityRegistrationData data = new ActivityRegistrationData();
                            data.setRegisteredCount(userList.size());
                            data.setMaleRegisteredCount((int) maleCount);
                            data.setFemaleRegisteredCount((int) femaleCount);
                            return data;
                        }
                ));
    }

    private String getAppleProductId(BigDecimal price) {
        ActivityPayConfigVo activityPayConfig = getPayConfig();
        if (Objects.isNull(activityPayConfig) || !activityPayConfig.getIosPayTypes().contains(ActivityPayConfigVo.PayType.APPLE.getCode())) {
            return null;
        }
        if (price.compareTo(BigDecimal.ZERO) == 0) {
            return null;
        }
        if (MapUtils.isEmpty(activityPayConfig.getAppleProductIdMap())) {
            return null;
        }
        String appleProductId = activityPayConfig.getAppleProductIdMap().get(price);
        if (StringUtils.isBlank(appleProductId)) {
            log.error("未找到对应的苹果商品ID,price:{}", price);
            return null;
        }
        return appleProductId;
    }

}
