package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityOrganizer;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.ActivityRegistrationData;
import com.bxm.lovelink.common.dal.entity.ActivityType;
import com.bxm.lovelink.common.dal.entity.Region;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserOrder;
import com.bxm.lovelink.common.dal.entity.dao.activity.ActivityListDao;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityQueryDto;
import com.bxm.lovelink.common.dal.entity.dto.activity.ActivityListQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.activity.ActivityDetailVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.ActivityListVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.GuestRecommendVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.usermutualselection.ActivityUserVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.mapping.activity.ActivityMapping;
import com.bxm.lovelink.common.dal.service.IActivityOrganizerService;
import com.bxm.lovelink.common.dal.service.IActivityRegistrationService;
import com.bxm.lovelink.common.dal.service.IActivityService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.dal.service.IActivityTypeService;
import com.bxm.lovelink.common.dal.service.IRegionService;
import com.bxm.lovelink.common.dal.service.IUserBasicInfoService;
import com.bxm.lovelink.common.dal.service.IUserOrderService;
import com.bxm.lovelink.common.dal.service.IUserService;
import com.bxm.lovelink.constant.RedisKeys;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * <p>
 * 活动表 服务实现类
 * </p>
 *
 * @author dll
 * @since 2025-06-23
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class ActivityServiceImpl extends ServiceImpl<ActivityMapper, Activity> implements IActivityService {
    private final IActivityTypeService activityTypeService;
    private final IRegionService regionService;
    private final IActivityOrganizerService activityOrganizerService;
    private final IUserBasicInfoService userBasicInfoService;
    private final IUserService userService;
    private final IActivityRegistrationService activityRegistrationService;
    private final RedissonClient redissonClient;
    private final IUserOrderService userOrderService;
    @Override
    public List<ActivityListVo> registeredList(Long userId, ActivityListQueryDto dto) {
        List<ActivityListDao> list = baseMapper.registeredList(userId, dto, LocalDateTime.now());
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        return fillList(ActivityMapping.INSTANCE.daoToVoList(list));
    }

    @Override
    public Page<ActivityListVo> list(Long userId, ActivityQueryDto dto) {
        dto.setCurrent(Constants.ONE);
        dto.setSize(Integer.MAX_VALUE);
        List<ActivityType> activityTypeList = activityTypeService.list(new LambdaQueryWrapper<ActivityType>()
                .eq(Objects.nonNull(dto.getType()), ActivityType::getType, dto.getType()));
        if (CollectionUtils.isEmpty(activityTypeList)) {
            return new Page<>();
        }
        List<Long> visibleActivityIds = getVisibleActivityIds(userId);
        List<Integer> typeIdList = activityTypeList.stream().filter(Objects::nonNull).map(ActivityType::getId).collect(Collectors.toList());
        Page<ActivityListDao> page = baseMapper.list(userId, new Page<>(dto.getCurrent(), dto.getSize()), typeIdList, dto, visibleActivityIds, LocalDateTime.now());
        if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
            return new Page<>();
        }
        Page<ActivityListVo> voPage = ActivityMapping.INSTANCE.daoToVoPage(page);
        if (CollectionUtils.isEmpty(voPage.getRecords())) {
            return new Page<>();
        }
        List<ActivityListVo> records = voPage.getRecords();
        rebuildActivityListVo(records);
        rebuildStatus(records, userId);
        reSort(records);
        return voPage;
    }

    private void reSort(List<ActivityListVo> records) {
        for (ActivityListVo record : records) {
            Integer status = record.getStatus();
            boolean hasRegistered = record.getRegisterStatus() == Constants.ActivityRegistrationStatus.SUCCESS;

            int sortStatus;
            if (status == Constants.ActivityProgressStatus.REGISTERING) {
                sortStatus = hasRegistered ? Constants.ActivitySortStatus.REGISTERED : Constants.ActivitySortStatus.REGISTERING;
            } else if (status == Constants.ActivityProgressStatus.FULL) {
                sortStatus = Constants.ActivitySortStatus.FULL;
            } else if (status == Constants.ActivityProgressStatus.REGISTER_END) {
                sortStatus = hasRegistered ? Constants.ActivitySortStatus.REGISTERED : Constants.ActivitySortStatus.REGISTER_END;
            } else if (status == Constants.ActivityProgressStatus.IN_PROGRESS) {
                sortStatus = hasRegistered ? Constants.ActivitySortStatus.IN_PROGRESS : Constants.ActivitySortStatus.REGISTER_END;
            } else {
                sortStatus = Constants.ActivitySortStatus.END;
            }
            record.setSortStatus(sortStatus);
        }
        records.sort(Comparator
                .comparing(ActivityListVo::getSortStatus)
                .thenComparing(ActivityListVo::getActivityStartTime));    }

    private void rebuildStatus(List<ActivityListVo> records, Long userId) {
        UserBasicInfo userBasicInfo = userBasicInfoService.getOneByUserId(userId);
        if (Objects.isNull(userBasicInfo)) {
            return;
        }
        Integer gender = userBasicInfo.getGender();
        for (ActivityListVo activityListVo : records) {
            if (activityListVo.getRegisterEndTime().isBefore(LocalDateTime.now()) && activityListVo.getActivityStartTime().isAfter(LocalDateTime.now())) {
                activityListVo.setStatus(Constants.ActivityProgressStatus.REGISTER_END);
            }
            if (!activityListVo.getRegisterStartTime().isAfter(LocalDateTime.now()) && !activityListVo.getRegisterEndTime().isBefore(LocalDateTime.now())) {
                activityListVo.setStatus(Constants.ActivityProgressStatus.REGISTERING);
                if (!activityListVo.getRegisterStatus().equals(Constants.ActivityRegistrationStatus.SUCCESS)) {
                    Boolean full = isFull(activityListVo.getMaleParticipantsLimit(), activityListVo.getFemaleParticipantsLimit(), activityListVo.getMaleRegisteredCount(), activityListVo.getFemaleRegisteredCount(), gender);
                    if (full) {
                        activityListVo.setStatus(Constants.ActivityProgressStatus.FULL);
                    }
                }
            }
        }
    }

    private Boolean isFull(Integer maleParticipantsLimit, Integer femaleParticipantsLimit, Integer maleRegisteredCount, Integer femaleRegisteredCount, Integer gender) {
        if (Objects.isNull(gender)) {
            return (maleParticipantsLimit + femaleParticipantsLimit) <= (maleRegisteredCount + femaleRegisteredCount);
        }
        return gender == Constants.UserGender.MALE ? maleParticipantsLimit <= maleRegisteredCount : femaleParticipantsLimit <= femaleRegisteredCount;
    }


    private void rebuildActivityListVo(List<ActivityListVo> records) {
        List<Long> activityIds = records.stream().filter(Objects::nonNull).map(ActivityListVo::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(activityIds)) {
            return;
        }
        Map<Integer, Region> codeMap = getCodeMap(records);
        Map<Long, List<ActivityUserVo>> regiactUserMap = activityRegistrationService.getRegiactUserMap(activityIds);
        records.forEach(activityListVo -> {
            activityListVo.setActivityUserList(regiactUserMap.get(activityListVo.getId()));
            fillCodeName(activityListVo, codeMap);
        });
    }

    private void fillCodeName(ActivityListVo activityListVo, Map<Integer, Region> codeMap) {
        Integer code = Integer.parseInt(activityListVo.getCityCode());
        Region region = codeMap.get(code);
        if (region != null) {
            activityListVo.setCityName(region.getName());
        }
    }

    private static void fillCodeNameAndRegisteredCount(ActivityListVo activityListVo, Map<Integer, Region> codeMap, Map<Long, ActivityRegistrationData> registrationDataMap) {
        Integer code = Integer.parseInt(activityListVo.getCityCode());
        Region region = codeMap.get(code);
        if (region != null) {
            activityListVo.setCityName(region.getName());
        }
        ActivityRegistrationData activityRegistrationData = registrationDataMap.get(activityListVo.getId());
        if (activityRegistrationData != null) {
            activityListVo.setRegisteredCount(activityRegistrationData.getRegisteredCount());
            activityListVo.setMaleRegisteredCount(activityRegistrationData.getMaleRegisteredCount());
            activityListVo.setFemaleRegisteredCount(activityRegistrationData.getFemaleRegisteredCount());
        }
    }

    private Map<Integer, Region> getCodeMap(List<ActivityListVo> records) {
        Set<Integer> codeSet = records.stream().filter(Objects::nonNull).map(ActivityListVo::getCityCode).filter(Objects::nonNull).map(Integer::parseInt).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(codeSet)) {
            return Collections.emptyMap();
        }
        return Optional.ofNullable(regionService.listByIds(Lists.newArrayList(codeSet)))
                .filter(CollectionUtils::isNotEmpty)
                .map(list -> list.stream().collect(Collectors.toMap(Region::getCode, Function.identity(), (a, b) -> a)))
                .orElse(Collections.emptyMap());
    }

    @Override
    public List<ActivityListVo> endList(Long userId, ActivityListQueryDto dto) {
        List<Long> visibleActivityIds = Objects.isNull(userId) ? Collections.emptyList() : getVisibleActivityIds(userId);
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime threeMonthsAgo = now.minusMonths(3);
        List<Activity> activityList = baseMapper.selectList(
                new LambdaQueryWrapper<Activity>()
                        .le(Activity::getActivityEndTime, now)
                        .ge(Activity::getCreateTime, threeMonthsAgo)
                        .eq(Activity::getActivityStatus, Constants.ActivityStatus.OPEN)
                        .eq(StringUtils.isNotBlank(dto.getCityCode()), Activity::getCityCode, dto.getCityCode())
                        .and(wrapper -> wrapper
                                .eq(Activity::getOvertStatus, Constants.ActivityOvertStatus.OVERT)
                                .or(inner -> inner
                                        .eq(Activity::getOvertStatus, Constants.ActivityOvertStatus.NOT_OVERT)
                                        .in(CollectionUtils.isNotEmpty(visibleActivityIds), Activity::getId, visibleActivityIds)
                                )
                        )
                        .orderByDesc(Activity::getId)
        );
        return fillList(ActivityMapping.INSTANCE.toVoList(activityList));
    }

    private List<ActivityListVo> fillList(List<ActivityListVo> activityListVos) {
        List<Long> activityIds = activityListVos.stream().filter(Objects::nonNull).map(ActivityListVo::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(activityIds)) {
            return activityListVos;
        }
        Map<Integer, Region> codeMap = getCodeMap(activityListVos);
        activityListVos.forEach(vo -> {
            fillCodeName(vo, codeMap);
        });
        return activityListVos;
    }

    /**
     * 获取可见的活动id列表
     */
    private List<Long> getVisibleActivityIds(Long userId) {
        User user = userService.getById(userId);
        List<Long> visibleActivityIds = new ArrayList<>();
        if (StringUtils.isNotBlank(user.getUnionid())) {
            visibleActivityIds = baseMapper.selectVisibleActivityId(user.getUnionid());
        }
        if (StringUtils.isNotBlank(user.getMobile())) {
            // 通过手机号查询是否是活动负责人
            List<Long> organizerIds = activityOrganizerService.list(new LambdaQueryWrapper<ActivityOrganizer>()
                    .eq(ActivityOrganizer::getMobile, user.getMobile())).stream().filter(Objects::nonNull).map(ActivityOrganizer::getId).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(organizerIds)) {
                visibleActivityIds.addAll(baseMapper.selectActivityIdByOrganizerIds(organizerIds));
            }
        }
        return visibleActivityIds;
    }

    @Override
    public ActivityDetailVo detail(UserBasicInfo userBasicInfo, Long id) {
        Activity activity = getById(id);
        if (Objects.isNull(activity)) {
            return null;
        }
        ActivityDetailVo detailVo = ActivityMapping.INSTANCE.toDetailVo(activity);
        ActivityType activityType = activityTypeService.getById(activity.getActivityTypeId());
        if (Objects.nonNull(activityType)) {
            detailVo.setActivityBizType(activityType.getBizType());
        }
        Optional.ofNullable(detailVo.getCityCode())
                .map(Integer::parseInt)
                .map(regionService::getById)
                .map(Region::getName)
                .ifPresent(detailVo::setCityName);
        Optional.ofNullable(activity.getOrganizerId())
                .map(activityOrganizerService::getById)
                .map(ActivityOrganizer::getName)
                .ifPresent(detailVo::setOrganizerName);
        GuestRecommendVo guestRecommend = detailVo.getGuestRecommend();
        Integer gender = null;
        Long userId = null;
        if (Objects.nonNull(userBasicInfo)){
            gender = userBasicInfo.getGender();
            userId = userBasicInfo.getUserId();
        }
        if (Objects.nonNull(guestRecommend) && Objects.nonNull(gender)) {
            if (gender == Constants.UserGender.MALE) {
                buildGuestUser(guestRecommend.getFemaleGuestList());
            } else if (gender == Constants.UserGender.FEMALE) {
                buildGuestUser(guestRecommend.getMaleGuestList());
            }
        }
        Map<Long, List<ActivityUserVo>> regiactUserMap = activityRegistrationService.getRegiactUserMap(Lists.newArrayList(id));
        if (MapUtils.isNotEmpty(regiactUserMap)) {
            detailVo.setActivityUserList(regiactUserMap.get(id));
        }
        LocalDateTime now = LocalDateTime.now();
        if (now.isBefore(activity.getActivityStartTime())) {
            detailVo.setStatus(Constants.ActivityProgressStatus.NOT_START);
        }
        if (now.isAfter(activity.getActivityEndTime())) {
            detailVo.setStatus(Constants.ActivityProgressStatus.END);
        }
        if (!now.isBefore(activity.getActivityStartTime())
                && !now.isAfter(activity.getActivityEndTime())) {
            detailVo.setStatus(Constants.ActivityProgressStatus.IN_PROGRESS);
        }
        if (!activity.getRegisterStartTime().isAfter(LocalDateTime.now()) && !activity.getRegisterEndTime().isBefore(LocalDateTime.now())) {
            detailVo.setStatus(Constants.ActivityProgressStatus.REGISTERING);
        }

        // 计算活动报名价格
        BigDecimal activityPrice = activityRegistrationService.calculateActivityPrice(activity, userId);
        BigDecimal activityShowPrice = calculateActivityShowPrice(activity);
        detailVo.setActivityPrice(activityPrice);
        detailVo.setActivityShowPrice(activityShowPrice);
        // 已报名
        if (Objects.nonNull(userId)) {
            ActivityRegistration activityRegistration = activityRegistrationService.getOne(new LambdaQueryWrapper<ActivityRegistration>()
                    .eq(ActivityRegistration::getActivityId, id)
                    .eq(ActivityRegistration::getUserId, userBasicInfo.getUserId())
                    .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS));
            if (Objects.nonNull(activityRegistration)) {
                detailVo.setRegisterStatus(Constants.ActivityRegistrationStatus.SUCCESS);
                return detailVo;
            }
        }
        // 没报名
        if (activity.getActivityStartTime().isAfter(LocalDateTime.now())) {
            if (activity.getRegisterEndTime().isBefore(LocalDateTime.now()) && activity.getActivityStartTime().isAfter(LocalDateTime.now())) {
                detailVo.setStatus(Constants.ActivityProgressStatus.REGISTER_END);
            }
            Boolean full = isFull(activity.getMaleParticipantsLimit(), activity.getFemaleParticipantsLimit(), detailVo.getMaleRegisteredCount(), detailVo.getFemaleRegisteredCount(), gender);
            if (full) {
                detailVo.setStatus(Constants.ActivityProgressStatus.FULL);
            }
        }
        return detailVo;
    }

    private BigDecimal calculateActivityShowPrice(Activity activity) {
        if (Objects.isNull(activity.getRegisterPrice()) || activity.getRegisterPrice().compareTo(BigDecimal.ZERO) <= 0) {
            return BigDecimal.ZERO;
        }
        if (Objects.nonNull(activity.getDiscountPrice()) && Objects.nonNull(activity.getDiscountLimitType()) && activity.isDiscountApplied()) {
            return activity.getDiscountPrice();
        }
        return activity.getRegisterPrice();
    }

    @Override
    public List<ActivityListVo> myList(Long userId, ActivityListQueryDto dto) {
        List<ActivityListDao> list = baseMapper.myList(userId, dto, LocalDateTime.now());
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        List<ActivityListVo> activityListVos = ActivityMapping.INSTANCE.daoToVoList(list);
        if (CollectionUtils.isEmpty(activityListVos)) {
            return new ArrayList<>();
        }
        rebuildActivityListVo(activityListVos);
        return activityListVos;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void releaseStock(String orderNum) {
        RLock lock = redissonClient.getLock(RedisKeys.activityOrderTimeoutLock(orderNum).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            return;
        }
        try {
            // 通过订单号查询活动报名信息
            ActivityRegistration activityRegistration = activityRegistrationService.getOneByOrderNo(orderNum);
            if (Objects.isNull(activityRegistration)) {
                return;
            }
            UserOrder userOrder = userOrderService.query(orderNum);
            if (userOrder.getStatus() >= Constants.UserOrder.STATUS_SUCCESS) {
                return;
            }
            Long activityId = activityRegistration.getActivityId();
            // 释放早鸟票名额
            if (Objects.equals(Constants.YES, activityRegistration.getDiscountApplied())) {
                baseMapper.releaseDiscountRegisterCount(activityId);
                log.info("释放早鸟票名额,activityId:{},userId:{}", activityId, activityRegistration.getUserId());
            }
            Integer gender = activityRegistration.getGender();
            if (Constants.UserGender.MALE == gender) {
                baseMapper.releaseMaleRegisterCount(activityId);
                log.info("释放男票名额,activityId:{},userId:{}", activityId, activityRegistration.getUserId());
            } else if (Constants.UserGender.FEMALE == gender) {
                baseMapper.releaseFemaleRegisterCount(activityId);
                log.info("释放女票名额,activityId:{},userId:{}", activityId, activityRegistration.getUserId());
            }
            activityRegistration.setStatus(Constants.ActivityRegistrationStatus.TIMEOUT_NOT_PAY);
            activityRegistration.setModifyTime(LocalDateTime.now());
            activityRegistrationService.updateById(activityRegistration);

            userOrderService.update(new LambdaUpdateWrapper<UserOrder>()
                    .set(UserOrder::getStatus, Constants.UserOrder.STATUS_CLOSED)
                    .eq(UserOrder::getOrderNum, orderNum));
        } finally {
            lock.unlock();
        }
    }

    @Override
    public Page<ActivityListVo> listWithOutLogin(ActivityQueryDto dto) {
        dto.setCurrent(Constants.ONE);
        dto.setSize(Integer.MAX_VALUE);
        List<ActivityType> activityTypeList = activityTypeService.list(new LambdaQueryWrapper<ActivityType>()
                .eq(Objects.nonNull(dto.getType()), ActivityType::getType, dto.getType()));
        if (CollectionUtils.isEmpty(activityTypeList)) {
            return new Page<>();
        }
        List<Integer> typeIdList = activityTypeList.stream().filter(Objects::nonNull).map(ActivityType::getId).collect(Collectors.toList());
        Page<ActivityListDao> page = baseMapper.listWithOutLogin(new Page<>(dto.getCurrent(), dto.getSize()), typeIdList, dto, LocalDateTime.now());
        if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
            return new Page<>();
        }
        Page<ActivityListVo> voPage = ActivityMapping.INSTANCE.daoToVoPage(page);
        if (CollectionUtils.isEmpty(voPage.getRecords())) {
            return new Page<>();
        }
        List<ActivityListVo> records = voPage.getRecords();
        rebuildActivityListVo(records);
        rebuildStatus(records);
        reSort(records);
        return voPage;
    }

    private void rebuildStatus(List<ActivityListVo> records) {
        records.forEach(dao -> {
            if (dao.getRegisterEndTime().isBefore(LocalDateTime.now()) && dao.getActivityStartTime().isAfter(LocalDateTime.now())) {
                dao.setStatus(Constants.ActivityProgressStatus.REGISTER_END);
            }
            if (!dao.getRegisterStartTime().isAfter(LocalDateTime.now()) && !dao.getRegisterEndTime().isBefore(LocalDateTime.now())) {
                dao.setStatus(Constants.ActivityProgressStatus.REGISTERING);
                Boolean full = isFull(dao.getMaleParticipantsLimit(), dao.getFemaleParticipantsLimit(), dao.getMaleRegisteredCount(), dao.getFemaleRegisteredCount(), null);
                if (full) {
                    dao.setStatus(Constants.ActivityProgressStatus.FULL);
                }
            }
        });
    }

    private void buildGuestUser(List<GuestRecommendVo.GuestUser> guestUserList) {
        guestUserList.removeIf(guestUser -> Objects.isNull(guestUser) || Objects.isNull(guestUser.getUserId()));
        List<Long> userIdList = guestUserList.stream().filter(Objects::nonNull).map(GuestRecommendVo.GuestUser::getUserId).filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userIdList)) {
            return;
        }
        List<User> userList = userService.listByIds(userIdList);
        if (CollectionUtils.isEmpty(userList)) {
            return;
        }
        Map<Long, User> userMap = userList.stream().collect(Collectors.toMap(User::getId, user -> user));
        List<UserBasicInfo> userBasicInfoList = userBasicInfoService.list(new LambdaQueryWrapper<UserBasicInfo>().in(UserBasicInfo::getUserId, userMap.keySet()));
        if (CollectionUtils.isEmpty(userBasicInfoList)) {
            return;
        }
        Map<Long, UserBasicInfo> userBasicInfoMap = userBasicInfoList.stream().collect(Collectors.toMap(UserBasicInfo::getUserId, user -> user));
        for (GuestRecommendVo.GuestUser guestUser : guestUserList) {
            User user = userMap.get(guestUser.getUserId());
            UserBasicInfo basicInfo = userBasicInfoMap.get(user.getId());
            guestUser.setUserNumber(user.getUserNumber());
            guestUser.setNickname(basicInfo.getNickname());
            guestUser.setAvatar(basicInfo.getAvatar());
            if (Objects.nonNull(basicInfo.getBirthday())) {
                guestUser.setAge(basicInfo.getBirthday().until(LocalDate.now()).getYears());
            }
            guestUser.setJob(basicInfo.getJob());
            guestUser.setRuleTag(basicInfo.getRuleTag());
            guestUser.setHeight(basicInfo.getHeight());
        }
    }
}
