package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.contant.UserConstants;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.ActivityUserMutualSelection;
import com.bxm.lovelink.common.dal.entity.ActivityUserSelection;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserComposite;
import com.bxm.lovelink.common.dal.entity.dao.activity.ActivityUserDao;
import com.bxm.lovelink.common.dal.entity.dto.activity.usermutualselection.ActivityMutualSelectionQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.activity.usermutualselection.ActivityUserSelectionResultVo;
import com.bxm.lovelink.common.dal.entity.vo.activity.usermutualselection.ActivityUserSelectionStatisticsVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserBasicInfoVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeSimpleVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.mapper.ActivityRegistrationMapper;
import com.bxm.lovelink.common.dal.mapper.ActivityUserMutualSelectionMapper;
import com.bxm.lovelink.common.dal.mapper.UserBasicInfoMapper;
import com.bxm.lovelink.common.dal.mapper.UserMapper;
import com.bxm.lovelink.common.dal.mapping.UserInfoMapping;
import com.bxm.lovelink.common.dal.mapping.activity.ActivityUserMutualSelectionMapping;
import com.bxm.lovelink.common.dal.service.IActivityUserMutualSelectionService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.dal.service.IActivityUserSelectionService;
import com.bxm.lovelink.common.dal.service.IUserBasicInfoService;
import com.bxm.lovelink.common.utils.GeoDistanceUtils;
import com.bxm.warcar.integration.pair.Pair;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>
 * 活动用户互选成功表 服务实现类
 * </p>
 *
 * @author dll
 * @since 2025-06-23
 */
@Service
@RequiredArgsConstructor
public class ActivityUserMutualSelectionServiceImpl extends ServiceImpl<ActivityUserMutualSelectionMapper, ActivityUserMutualSelection> implements IActivityUserMutualSelectionService {
    private final ActivityRegistrationMapper activityRegistrationMapper;
    private final IUserBasicInfoService userBasicInfoService;
    private final IActivityUserSelectionService activityUserSelectionService;
    private final ActivityMapper activityMapper;
    private final Pair pair;
    private final UserBasicInfoMapper userBasicInfoMapper;
    private final UserMapper userMapper;

    @Override
    public List<UserCompositeSimpleVo> userList(ActivityMutualSelectionQueryDto dto, UserBasicInfo userBasicInfo) {
        if (Objects.isNull(dto) || Objects.isNull(userBasicInfo)) {
            return null;
        }
        List<UserComposite> list = userBasicInfoMapper.selectActivityUserList(dto, userBasicInfo.getReverseGender(), userBasicInfo.getUserId());
        List<UserCompositeVo> userCompositeVos = UserInfoMapping.INSTANCE.doToVo(list);
        userCompositeVos.forEach(vo -> {
            if (Objects.nonNull(vo)){
                UserBasicInfoVo basicInfo = vo.getUserBasicInfo();
                Double distance = GeoDistanceUtils.calculateDistance(userBasicInfo.getLatitude(), userBasicInfo.getLongitude(), basicInfo.getLatitude(), basicInfo.getLongitude());
                basicInfo.setDistance(distance);
                basicInfo.setSchoolName(Objects.equals(basicInfo.getSchoolHide(), Constants.NO) ? null : basicInfo.getSchoolName());
            }
        });
        return UserInfoMapping.INSTANCE.toList(userCompositeVos);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<ActivityUserSelectionResultVo> selectResult(Long activityId, UserBasicInfo userBasicInfo, User currentUser) {
        Activity activity = activityMapper.selectById(activityId);
        if (Objects.isNull(activity)) {
            throw new IllegalStateException("活动不存在");
        }
        if (!activity.getActivityEndTime().isBefore(LocalDateTime.now())) {
            throw new IllegalStateException("活动未结束，无法查看互选结果");
        }
        Long currentUserId = userBasicInfo.getUserId();
        boolean isMale = userBasicInfo.getGender().equals(Constants.UserGender.MALE);
        List<ActivityUserSelection> userSelections = activityUserSelectionService.list(new LambdaQueryWrapper<ActivityUserSelection>()
                .eq(ActivityUserSelection::getActivityId, activityId)
                .eq(ActivityUserSelection::getUserId, currentUserId)
        );
        if (CollectionUtils.isEmpty(userSelections)) {
            return Collections.emptyList();
        }
        List<ActivityUserMutualSelection> mutualSelections = list(
                new LambdaQueryWrapper<ActivityUserMutualSelection>()
                        .eq(ActivityUserMutualSelection::getActivityId, activityId)
                        .eq(isMale ? ActivityUserMutualSelection::getMaleUserId : ActivityUserMutualSelection::getFemaleUserId, currentUserId)
        );
        Map<String, ActivityUserMutualSelection> mutualMap = mutualSelections.stream()
                .collect(Collectors.toMap(
                        sel -> sel.getMaleUserId() + "_" + sel.getFemaleUserId(),
                        Function.identity(),
                        (a, b) -> a
                ));
        List<Long> targetUserIds = userSelections.stream()
                .filter(Objects::nonNull)
                .map(ActivityUserSelection::getTargetUserId)
                .collect(Collectors.toList());
        Map<Long, UserBasicInfo> targetUserInfoMap = userBasicInfoService.listIncludeDeleted(targetUserIds).stream()
                .collect(Collectors.toMap(UserBasicInfo::getUserId, Function.identity()));
        Map<Long, User> userMap = userMapper.selectBatchIds(targetUserIds).stream()
                .collect(Collectors.toMap(User::getId, Function.identity()));
        List<ActivityUserSelectionResultVo> userSelectionResultVos = ActivityUserMutualSelectionMapping.INSTANCE.toVoList(userSelections);
        for (ActivityUserSelectionResultVo userSelection : userSelectionResultVos) {
            userSelection.setNickname(userBasicInfo.getNickname());
            userSelection.setAvatar(userBasicInfo.getAvatar());
            userSelection.setUserNumber(currentUser.getUserNumber());
            Long targetUserId = userSelection.getTargetUserId();
            UserBasicInfo targetInfo = targetUserInfoMap.get(targetUserId);
            if (targetInfo == null) {
                continue;
            }
            User user = userMap.get(targetUserId);
            if (user == null){
                continue;
            }
            userSelection.setTargetUserNumber(user.getUserNumber());
            userSelection.setTargetUserId(targetUserId);
            userSelection.setTargetUserNickname(targetInfo.getNickname());
            userSelection.setTargetUserAvatar(targetInfo.getAvatar());
            Long maleId = isMale ? currentUserId : targetUserId;
            Long femaleId = isMale ? targetUserId : currentUserId;
            String key = maleId + "_" + femaleId;
            if (mutualMap.containsKey(key)) {
                userSelection.setSelectionResult(Constants.YES);
            } else {
                // 查询是否对方也选了我
                boolean mutualSelected = activityUserSelectionService.count(
                        new LambdaQueryWrapper<ActivityUserSelection>()
                                .eq(ActivityUserSelection::getActivityId, activityId)
                                .eq(ActivityUserSelection::getUserId, targetUserId)
                                .eq(ActivityUserSelection::getTargetUserId, currentUserId)
                ) > 0;

                if (mutualSelected) {
                    userSelection.setSelectionResult(Constants.YES);
                    // 补存互选记录
                    ActivityUserMutualSelection mutual = new ActivityUserMutualSelection();
                    mutual.setActivityId(activityId);
                    mutual.setMaleUserId(maleId);
                    mutual.setFemaleUserId(femaleId);
                    save(mutual);
                } else {
                    userSelection.setSelectionResult(Constants.NO);
                }
            }
        }
        return userSelectionResultVos;
    }

    @Override
    public Boolean canFreeRegister(Long activityId, UserBasicInfo userBasicInfo) {
        Long userId = userBasicInfo.getUserId();
        Activity activity = activityMapper.selectById(activityId);
        if (activity == null) {
            return false;
        }
        if (!activityTypeFree(activity)) {
            return false;
        }
        // 判断是否已经领取过这个活动的免费机会
        if (hasReceivedFree(userId, activityId, activity.getActivityTypeId())) {
            return false;
        }
        if (hasMutualSelection(userId, activityId, userBasicInfo.getGender())) {
            return false;
        }
        // 判断用户参与本次活动是否付费
        ActivityRegistration activityRegistration = getSuccessRegistration(userId, activityId);
        return Objects.nonNull(activityRegistration) && StringUtils.isNotBlank(activityRegistration.getOrderNo());
    }

    private Boolean activityTypeFree(Activity activity) {
        Set<String> activityTypeIdSet = pair.get(Constants.PairKey.FREE_REGISTRATION_ACTIVITY_TYPE_ID).ofHashSet();
        if (CollectionUtils.isEmpty(activityTypeIdSet)) {
            return false;
        }
        return activityTypeIdSet.contains(String.valueOf(activity.getActivityTypeId()));
    }

    @Override
    public void receiveFreeRegister(Long activityId, UserBasicInfo userBasicInfo) {
        Long userId = userBasicInfo.getUserId();
        Activity activity = activityMapper.selectById(activityId);
        if (activity == null) {
            throw new IllegalStateException("活动不存在");
        }
        if (!activityTypeFree(activity)) {
            throw new IllegalStateException("当前活动类型不支持领取免费报名机会");
        }
        if (hasReceivedFree(userId, activityId, activity.getActivityTypeId())) {
            throw new IllegalStateException("已经领取过免费机会了");
        }
        if (!activity.getActivityEndTime().isBefore(LocalDateTime.now())) {
            throw new IllegalStateException("活动未结束，无法领取");
        }
        if (hasMutualSelection(userId, activityId, userBasicInfo.getGender())) {
            throw new IllegalStateException("存在互选成功结果无法领取");
        }
        ActivityRegistration activityRegistration = getSuccessRegistration(userId, activityId);
        if (Objects.isNull(activityRegistration) || StringUtils.isBlank(activityRegistration.getOrderNo())) {
            throw new IllegalStateException("非付费报名无法领取");
        }
        // 领取免费机会
        activityRegistration.setFreeActivityTypeId(activity.getActivityTypeId());
        activityRegistrationMapper.updateById(activityRegistration);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void generateMutualSelectResult(Long activityId) {
        List<ActivityUserSelection> userSelections = activityUserSelectionService.list(new LambdaQueryWrapper<ActivityUserSelection>()
                .eq(ActivityUserSelection::getActivityId, activityId));
        if (CollectionUtils.isEmpty(userSelections)) {
            return;
        }
        Set<String> existingMutualKeys = list(new LambdaQueryWrapper<ActivityUserMutualSelection>()
                .eq(ActivityUserMutualSelection::getActivityId, activityId))
                .stream()
                .map(mutual -> mutual.getMaleUserId() + "_" + mutual.getFemaleUserId())
                .collect(Collectors.toSet());
        Set<Long> allUserIds = userSelections.stream()
                .flatMap(s -> Stream.of(s.getUserId(), s.getTargetUserId()))
                .collect(Collectors.toSet());
        Map<Long, UserBasicInfo> userInfoMap = userBasicInfoService.listIncludeDeleted(new ArrayList<>(allUserIds))
                .stream().collect(Collectors.toMap(UserBasicInfo::getUserId, Function.identity()));
        Map<Long, Set<Long>> selectionMap = new HashMap<>();
        List<ActivityUserMutualSelection> mutualInsertList = new ArrayList<>();
        for (ActivityUserSelection userSelection : userSelections) {
            selectionMap.computeIfAbsent(userSelection.getUserId(), k -> new HashSet<>()).add(userSelection.getTargetUserId());

            Long userId = userSelection.getUserId();
            Long targetUserId = userSelection.getTargetUserId();
            UserBasicInfo userInfo = userInfoMap.get(userId);
            if (Objects.isNull(userInfo)) {
                continue;
            }
            boolean isMale = userInfo.getGender().equals(Constants.UserGender.MALE);
            Long maleId = isMale ? userId : targetUserId;
            Long femaleId = isMale ? targetUserId : userId;

            String mutualKey = maleId + "_" + femaleId;
            if (existingMutualKeys.contains(mutualKey)) {
                continue;
            }
            if (selectionMap.getOrDefault(targetUserId, Collections.emptySet()).contains(userId)) {
                ActivityUserMutualSelection mutual = new ActivityUserMutualSelection();
                mutual.setActivityId(activityId);
                mutual.setMaleUserId(maleId);
                mutual.setFemaleUserId(femaleId);
                mutualInsertList.add(mutual);
                existingMutualKeys.add(mutualKey);
            }
        }
        if (CollectionUtils.isNotEmpty(mutualInsertList)) {
            saveBatch(mutualInsertList);
        }

    }

    @Override
    public boolean exists(Long maleUserId, Long femaleUserId) {
        LambdaQueryWrapper<ActivityUserMutualSelection> wrapper = new LambdaQueryWrapper<ActivityUserMutualSelection>()
                .eq(ActivityUserMutualSelection::getMaleUserId, maleUserId)
                .eq(ActivityUserMutualSelection::getFemaleUserId, femaleUserId);
        return count(wrapper) > 0;
    }

    @Override
    public void deleteUserMutualSelection(Long userId, Integer gender) {
        LambdaQueryWrapper<ActivityUserMutualSelection> queryWrapper = new LambdaQueryWrapper<>();
        if (Constants.UserGender.MALE == (gender)) {
            queryWrapper.eq(ActivityUserMutualSelection::getMaleUserId, userId);
        } else if (Constants.UserGender.FEMALE == (gender)) {
            queryWrapper.eq(ActivityUserMutualSelection::getFemaleUserId, userId);
        }
        remove(queryWrapper);
    }

    @Override
    public ActivityUserSelectionStatisticsVo statistics(ActivityMutualSelectionQueryDto dto, UserBasicInfo userBasicInfo) {
        ActivityUserSelectionStatisticsVo statisticsVo = new ActivityUserSelectionStatisticsVo();
        List<ActivityUserDao> list = activityRegistrationMapper.selectActivityUserList(new ActivityMutualSelectionQueryDto().setActivityId(dto.getActivityId()), userBasicInfo.getReverseGender(), userBasicInfo.getUserId());
        List<ActivityUserDao> heartMeList = activityRegistrationMapper.selectActivityUserList(new ActivityMutualSelectionQueryDto().setActivityId(dto.getActivityId()).setTabType(ActivityMutualSelectionQueryDto.TabType.HEART_ME), userBasicInfo.getReverseGender(), userBasicInfo.getUserId());
        List<ActivityUserDao> myHeartList = activityRegistrationMapper.selectActivityUserList(new ActivityMutualSelectionQueryDto().setActivityId(dto.getActivityId()).setTabType(ActivityMutualSelectionQueryDto.TabType.MY_HEART), userBasicInfo.getReverseGender(), userBasicInfo.getUserId());
        statisticsVo.setTotalCount((long) list.size());
        statisticsVo.setHeartMeCount((long) heartMeList.size());
        statisticsVo.setMyHeartCount((long) myHeartList.size());
        if (Objects.isNull(dto.getTabType())) {
            return buildStatisticsVo(statisticsVo, list);
        }
        if (dto.getTabType().equals(ActivityMutualSelectionQueryDto.TabType.HEART_ME)) {
            return buildStatisticsVo(statisticsVo, heartMeList);
        }
        if (dto.getTabType().equals(ActivityMutualSelectionQueryDto.TabType.MY_HEART)) {
            return buildStatisticsVo(statisticsVo, myHeartList);
        }
        return statisticsVo;
    }

    private ActivityUserSelectionStatisticsVo buildStatisticsVo(ActivityUserSelectionStatisticsVo statisticsVo, List<ActivityUserDao> list) {
        if (CollectionUtils.isNotEmpty(list)) {
            long realNameCount = list.stream().filter(Objects::nonNull).filter(u -> u.getCertStatus().contains(UserConstants.CertTypeEnum.REAL_PEOPLE.getCode())).count();
            statisticsVo.setRealNameCount(realNameCount);
            long marriageCount = list.stream().filter(Objects::nonNull).filter(u -> u.getCertStatus().contains(UserConstants.CertTypeEnum.MARRIED.getCode())).count();
            statisticsVo.setMarriageCount(marriageCount);
            long educationCount = list.stream().filter(Objects::nonNull).filter(u -> u.getCertStatus().contains(UserConstants.CertTypeEnum.EDU.getCode())).count();
            statisticsVo.setEducationCount(educationCount);
            long carCount = list.stream().filter(Objects::nonNull).filter(u -> u.getCertStatus().contains(UserConstants.CertTypeEnum.CAR.getCode())).count();
            statisticsVo.setCarCount(carCount);
        }
        return statisticsVo;
    }

    private boolean hasMutualSelection(Long userId, Long activityId, Integer gender) {
        boolean isMale = gender.equals(Constants.UserGender.MALE);
        LambdaQueryWrapper<ActivityUserMutualSelection> wrapper = new LambdaQueryWrapper<ActivityUserMutualSelection>()
                .eq(ActivityUserMutualSelection::getActivityId, activityId)
                .eq(isMale ? ActivityUserMutualSelection::getMaleUserId : ActivityUserMutualSelection::getFemaleUserId, userId);
        return count(wrapper) > 0;
    }

    private boolean hasSelection(Long userId, Long activityId) {
       return activityUserSelectionService.count(new LambdaQueryWrapper<ActivityUserSelection>()
                .eq(ActivityUserSelection::getActivityId, activityId)
                .eq(ActivityUserSelection::getUserId, userId)) > 0;
    }

    private boolean hasReceivedFree(Long userId, Long activityId, Long activityTypeId) {
        return activityRegistrationMapper.selectCount(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getUserId, userId)
                .eq(ActivityRegistration::getActivityId, activityId)
                .eq(ActivityRegistration::getFreeActivityTypeId, activityTypeId)) > 0;
    }

    private ActivityRegistration getSuccessRegistration(Long userId, Long activityId) {
        return activityRegistrationMapper.selectOne(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getUserId, userId)
                .eq(ActivityRegistration::getActivityId, activityId)
                .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS));
    }
}
