package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.ActivityUserSelection;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.dto.activity.userselection.ActivityUserSelectionDto;
import com.bxm.lovelink.common.dal.entity.vo.activity.userselection.ActivityUserSelectionVo;
import com.bxm.lovelink.common.dal.mapper.ActivityMapper;
import com.bxm.lovelink.common.dal.mapper.ActivityRegistrationMapper;
import com.bxm.lovelink.common.dal.mapper.ActivityUserSelectionMapper;
import com.bxm.lovelink.common.dal.service.IActivityTypeService;
import com.bxm.lovelink.common.dal.service.IActivityUserSelectionService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.dal.service.IUserBasicInfoService;
import com.bxm.lovelink.constant.RedisKeys;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * <p>
 * 活动用户选择记录表 服务实现类
 * </p>
 *
 * @author dll
 * @since 2025-06-23
 */
@Service
@RequiredArgsConstructor
public class ActivityUserSelectionServiceImpl extends ServiceImpl<ActivityUserSelectionMapper, ActivityUserSelection> implements IActivityUserSelectionService {
    private final ActivityMapper activityMapper;
    private final ActivityRegistrationMapper activityRegistrationMapper;
    private final IUserBasicInfoService userBasicInfoService;
    private final IActivityTypeService activityTypeService;
    private final RedissonClient redissonClient;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(ActivityUserSelectionDto dto, UserBasicInfo userBasicInfo) {
        if (Objects.isNull(dto) || Objects.isNull(userBasicInfo)) {
            throw new IllegalArgumentException("参数不能为空");
        }
        Activity activity = activityMapper.selectById(dto.getActivityId());
        if (Objects.isNull(activity)) {
            throw new IllegalStateException("活动不存在");
        }
        // 判断活动是否支持互选
        Boolean supportMutualSelection = activityTypeService.supportMutualSelection(activity.getActivityTypeId());
        if (!supportMutualSelection) {
            throw new IllegalStateException("活动不支持互选");
        }
        if (activity.getActivityEndTime().isBefore(LocalDateTime.now())) {
            throw new IllegalStateException("活动已结束");
        }
        if (activity.getActivityStartTime().isAfter(LocalDateTime.now())) {
            throw new IllegalStateException("活动未开始");
        }
        ActivityRegistration activityRegistration = activityRegistrationMapper.selectOne(new LambdaQueryWrapper<ActivityRegistration>()
                .eq(ActivityRegistration::getActivityId, dto.getActivityId())
                .eq(ActivityRegistration::getUserId, userBasicInfo.getUserId())
                .eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS));
        if (activityRegistration == null) {
            throw new IllegalStateException("您未报名");
        }
        RLock lock = redissonClient.getLock(RedisKeys.activityUserSelectionLock(dto.getActivityId(), userBasicInfo.getUserId()).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            throw new IllegalStateException("正在选择中，请稍后再试");
        }
        try {
            Long currentUserId = userBasicInfo.getUserId();
            Integer gender = userBasicInfo.getGender();
            Long selectedUserId = dto.getSelectedUserId();

            UserBasicInfo selectedUser = userBasicInfoService.getOneByUserId(selectedUserId);
            if (Objects.isNull(selectedUser)) {
                throw new IllegalStateException("所选择的用户已注销");
            }
            if (Objects.equals(selectedUser.getGender(), gender)) {
                throw new IllegalStateException("不能选择同性别用户");
            }
            if (count(new LambdaQueryWrapper<ActivityUserSelection>()
                    .eq(ActivityUserSelection::getActivityId, dto.getActivityId())
                    .eq(ActivityUserSelection::getUserId, currentUserId)
                    .eq(ActivityUserSelection::getTargetUserId, selectedUserId)) > 0) {
                throw new IllegalStateException("已选择过该用户");
            }
            ActivityUserSelection selection = new ActivityUserSelection();
            selection.setActivityId(dto.getActivityId());
            selection.setUserId(currentUserId);
            selection.setTargetUserId(selectedUserId);
            save(selection);
            activityRegistration.setSignInStatus(Constants.ActivitySignStatus.SIGNED);
            activityRegistration.setSignInTime(LocalDateTime.now());
            activityRegistrationMapper.updateById(activityRegistration);
        } finally {
            lock.unlock();
        }

    }

    @Override
    public void deleteByActivityIdAndSelectedUserId(Long userId, ActivityUserSelectionDto dto) {
        ActivityUserSelection userSelection = getOne(new LambdaQueryWrapper<ActivityUserSelection>()
                .eq(ActivityUserSelection::getActivityId, dto.getActivityId())
                .eq(ActivityUserSelection::getUserId, userId)
                .eq(ActivityUserSelection::getTargetUserId, dto.getSelectedUserId()));
        if (Objects.isNull(userSelection)) {
            return;
        }
        removeById(userSelection);
    }

    @Override
    public List<ActivityUserSelectionVo> list(Long userId, Long activityId) {
        List<ActivityUserSelection> userSelections = list(new LambdaQueryWrapper<ActivityUserSelection>()
                .eq(ActivityUserSelection::getActivityId, activityId)
                .eq(ActivityUserSelection::getUserId, userId));
        if (CollectionUtils.isEmpty(userSelections)) {
            return Collections.emptyList();
        }
        List<Long> userIds = userSelections.stream().filter(Objects::nonNull).map(ActivityUserSelection::getTargetUserId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyList();
        }
        List<UserBasicInfo> userBasicInfos = userBasicInfoService.listIncludeDeleted(userIds);
        if (CollectionUtils.isEmpty(userBasicInfos)) {
            return Collections.emptyList();
        }
        Map<Long, UserBasicInfo> userBasicInfoMap = userBasicInfos.stream().filter(Objects::nonNull).collect(Collectors.toMap(UserBasicInfo::getUserId, user -> user));
        return userSelections.stream().map(userSelection -> {
            ActivityUserSelectionVo vo = new ActivityUserSelectionVo();
            vo.setUserId(userSelection.getTargetUserId());
            vo.setAvatar(userBasicInfoMap.get(userSelection.getTargetUserId()).getAvatar());
            return vo;
        }).collect(Collectors.toList());
    }
}
