package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.UserVisitRecord;
import com.bxm.lovelink.common.dal.entity.dto.heartbeat.UserClearUnReadDto;
import com.bxm.lovelink.common.dal.entity.vo.heartbeat.HeartBeatUnreadCountVo;
import com.bxm.lovelink.common.dal.mapper.UserVisitRecordMapper;
import com.bxm.lovelink.common.dal.service.IUserOnlineService;
import com.bxm.lovelink.common.dal.service.IUserRelationService;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.lovelink.common.contant.UserConstants;
import com.bxm.lovelink.common.dal.entity.InviteConfig;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserOtherInfo;
import com.bxm.lovelink.common.dal.entity.UserRelation;
import com.bxm.lovelink.common.dal.entity.dao.heartbeat.HeartBeatLisDao;
import com.bxm.lovelink.common.dal.entity.dto.heartbeat.HeartBeatDto;
import com.bxm.lovelink.common.dal.entity.dto.heartbeat.HeartBeatQueryDto;
import com.bxm.lovelink.common.dal.entity.dto.heartbeat.HeartBeatReadDto;
import com.bxm.lovelink.common.dal.entity.vo.heartbeat.HeartBeatListVo;
import com.bxm.lovelink.common.dal.entity.vo.heartbeat.HeartBeatResultVo;
import com.bxm.lovelink.common.dal.mapper.UserRelationMapper;
import com.bxm.lovelink.common.dal.mapping.HeartBeatMapping;
import com.bxm.lovelink.common.dal.service.HeartBeatService;
import com.bxm.lovelink.common.dal.service.IRegionService;
import com.bxm.lovelink.common.dal.service.IUserOtherInfoService;
import com.bxm.warcar.cache.Counter;
import com.bxm.lovelink.common.dal.service.ISwipeControlService;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 心动管理服务实现类
 * </p>
 *
 * @author weixing
 */
@Slf4j
@Service
public class HeartBeatServiceImpl implements HeartBeatService {
    private final UserRelationMapper userRelationMapper;

    private final IUserOtherInfoService userOtherInfoService;

    private final IRegionService regionService;

    private final Counter counter;
    private final Pair pair;
    private final ISwipeControlService swipeControlService;
    private final IUserOnlineService userOnlineService;
    private final UserVisitRecordMapper userVisitRecordMapper;
    private final IUserRelationService userRelationService;

    public HeartBeatServiceImpl(UserRelationMapper userRelationMapper, IUserOtherInfoService userOtherInfoService, IRegionService regionService, Counter counter, Pair pair, ISwipeControlService swipeControlService, IUserOnlineService userOnlineService, UserVisitRecordMapper userVisitRecordMapper, IUserRelationService userRelationService) {
        this.userRelationMapper = userRelationMapper;
        this.userOtherInfoService = userOtherInfoService;
        this.regionService = regionService;
        this.counter = counter;
        this.pair = pair;
        this.swipeControlService = swipeControlService;
        this.userOnlineService = userOnlineService;
        this.userVisitRecordMapper = userVisitRecordMapper;
        this.userRelationService = userRelationService;
    }


    @Override
    public IPage<HeartBeatListVo> queryMyHeartBeatList(HeartBeatQueryDto dto) {
        validateLocationIfRequired(dto);
        int total = userRelationMapper.countMyHeartBeatList(dto);
        if (total == 0) {
            return new Page<>(dto.getCurrent(), dto.getSize());
        }
        int offset = (dto.getCurrent() - 1) * dto.getSize();
        List<HeartBeatLisDao> list = userRelationMapper.queryMyHeartBeatList(dto, offset, dto.getSize());
        return getHeartBeatListVoPage(dto, total, list);
    }

    /**
     * 填充城市名称
     */
    private void fillCityName(Page<HeartBeatListVo> page) {
        List<Integer> cityCodeList = page.getRecords().stream()
                .map(HeartBeatListVo::getCurrentPlaceCity)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(cityCodeList)) {
            return;
        }
        Map<Integer, String> codeNameMap = regionService.queryByCodes(cityCodeList);
        if (MapUtils.isEmpty(codeNameMap)) {
            return;
        }
        page.getRecords().forEach(vo -> {
            Integer cityCode = vo.getCurrentPlaceCity();
            if (cityCode != null) {
                vo.setCurrentPlaceCityName(codeNameMap.get(cityCode));
            }
        });
    }

    @Override
    public IPage<HeartBeatListVo> queryHeartBeatMeList(HeartBeatQueryDto dto) {
        validateLocationIfRequired(dto);
        int total = userRelationMapper.countHeartBeatMeList(dto);
        if (total == 0) {
            return new Page<>(dto.getCurrent(), dto.getSize());
        }
        int offset = (dto.getCurrent() - 1) * dto.getSize();
        List<HeartBeatLisDao> list = userRelationMapper.queryHeartBeatMeList(dto, offset, dto.getSize());
        return getHeartBeatListVoPage(dto, total, list);
    }

    private IPage<HeartBeatListVo> getHeartBeatListVoPage(HeartBeatQueryDto dto, int total, List<HeartBeatLisDao> list) {
        Page<HeartBeatLisDao> heartBeatLisDaoPage = new Page<>(dto.getCurrent(), dto.getSize(), total);
        heartBeatLisDaoPage.setRecords(list);
        Page<HeartBeatListVo> page = HeartBeatMapping.INSTANCE.toPage(heartBeatLisDaoPage);
        if (CollectionUtils.isEmpty(page.getRecords())) {
            return page;
        }
        fillCityName(page);
        fillOnlineStatus(page.getRecords());
        return page;
    }

    private void fillOnlineStatus(List<HeartBeatListVo> records) {
        List<Long> userIds = records.stream().filter(Objects::nonNull).map(HeartBeatListVo::getUserId).collect(Collectors.toList());
        Map<Long, LocalDateTime> onlineMap = userOnlineService.getBatchLastOnlineTime(userIds);
        if (MapUtils.isEmpty(onlineMap)) {
            return;
        }
        for (HeartBeatListVo record : records) {
            LocalDateTime lastOnlineTime = onlineMap.get(record.getUserId());
            /**
             *  1. “当前在线”：最后一次在线是否20分钟内
             *  2. “26分钟前在线”：在线时间小于1小时，向下取整分钟数
             *  3. “2小时前在线”：在线时间大于1小时，向下取整小时数
             *  4. “2天前在线”：在线时间大于24小时，向下取整天数
             */
            if (Objects.nonNull(lastOnlineTime)) {
                record.setOnlineStatus(getOnlineStatus(lastOnlineTime));
            }
        }
    }

    private static String getOnlineStatus(LocalDateTime lastOnlineTime) {
        Duration duration = Duration.between(lastOnlineTime, LocalDateTime.now());
        long minutes = duration.toMinutes();
        long hours = duration.toHours();
        long days = duration.toDays();
        if (minutes <= 20) {
            return "当前在线";
        } else if (hours < 1) {
            return minutes + "分钟前在线";
        } else if (hours < 24) {
            return hours + "小时前在线";
        } else {
            return days + "天前在线";
        }
    }

    private void validateLocationIfRequired(HeartBeatQueryDto dto) {
        if (StringUtils.equals(dto.getOrderBy(), HeartBeatQueryDto.OrderByEnum.DISTANCE.getValue())) {
            if (dto.getLat() == null || dto.getLon() == null) {
                throw new IllegalArgumentException("经纬度不能为空");
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public HeartBeatResultVo swipe(UserBasicInfo userBasicInfo, HeartBeatDto heartBeatDto) {
        HeartBeatResultVo resultVo = new HeartBeatResultVo();
        log.info("heartBeatDto userId：{}，heartBeatDto:{}", userBasicInfo.getUserId(), JsonHelper.convert(heartBeatDto));
        Long userId = userBasicInfo.getUserId();
        Objects.requireNonNull(userId, "用户Id不能为空");

        Long targetUserId = heartBeatDto.getTargetUserId();
        Integer relationType = heartBeatDto.getType();

        // 滑动限制
        List<Integer> certStatus = userBasicInfo.getCertStatus();
        boolean isRealPeople = !CollectionUtils.isEmpty(certStatus) && certStatus.contains(UserConstants.CertTypeEnum.REAL_PEOPLE.getCode());
        if (heartBeatDto.getChannel().equals(Constants.HeartBeatChannel.HOME)) {
            Integer swipeResult = allowSwipe(userId, isRealPeople);
            if (!isRealPeople) {
                if (swipeResult == Constants.SwipeLimitStatus.OUT_OF_LIMIT) {
                    return resultVo.setReachedRealNameLimit(true).setAllowSwipe(false);
                }
                if (relationType.equals(Constants.UserRelationType.HEARTBEAT.getType())) {
                    resultVo.setReachedRealNameLimit(true);
                }
            }
        }
        // 查询当前用户的资料引导状态
        int heartBeatStatus = Constants.YES;
        if (Constants.UserRelationType.HEARTBEAT.getType().equals(relationType)) {
            UserOtherInfo userOtherInfo = userOtherInfoService.getOneByUserId(userId);
            heartBeatStatus = Optional.ofNullable(userOtherInfo)
                    .map(UserOtherInfo::getBasicInfoGuideFinished)
                    .orElse(Constants.NO);
        }

        // 查询现有关系
        UserRelation myRelation = userRelationMapper.selectOne(
                new LambdaQueryWrapper<UserRelation>()
                        .eq(UserRelation::getUserId, userId)
                        .eq(UserRelation::getTargetUserId, targetUserId)
                        .eq(UserRelation::getDeleted, Constants.NOT_DELETED)
        );
        UserRelation hisRelation = userRelationMapper.selectOne(
                new LambdaQueryWrapper<UserRelation>()
                        .eq(UserRelation::getUserId, targetUserId)
                        .eq(UserRelation::getTargetUserId, userId)
                        .eq(UserRelation::getDeleted, Constants.NOT_DELETED)
        );
        if (myRelation == null && hisRelation == null) {
            return handleBothNone(userId, targetUserId, relationType, resultVo, heartBeatStatus);
        } else if (myRelation == null) {
            return handleMyNoneHisExist(userId, targetUserId, relationType, hisRelation, resultVo, heartBeatStatus);
        } else if (hisRelation == null) {
            return handleMyExistHisNone(userId, targetUserId, relationType, myRelation, resultVo, heartBeatStatus);
        } else {
            return handleBothExist(userId, targetUserId, relationType, myRelation, hisRelation, resultVo, heartBeatStatus);
        }
    }

    private HeartBeatResultVo handleBothExist(Long userId, Long targetUserId, Integer relationType, UserRelation myRelation, UserRelation hisRelation, HeartBeatResultVo resultVo, int heartBeatStatus) {
        if (relationType.equals(Constants.UserRelationType.HEARTBEAT.getType())) {
            if (Constants.UserRelationType.DISLIKE.getType().equals(myRelation.getType())) {
                boolean matched = Constants.UserRelationType.HEARTBEAT.getType().equals(hisRelation.getType());
                updateRelation(myRelation, relationType, matched, heartBeatStatus);
                incrBeLoved(targetUserId, heartBeatStatus);
                if (matched && Constants.NO == hisRelation.getMatched()) {
                    updateMatched(hisRelation, true);
                }
                return resultVo.setMatched(matched);
            } else if (Constants.UserRelationType.HEARTBEAT.getType().equals(myRelation.getType())) {
                if (Constants.UserRelationType.HEARTBEAT.getType().equals(hisRelation.getType())) {
                    if (Constants.YES == myRelation.getMatched()) {
                        // 已经匹配过不再提示
                        return resultVo.setBeforeMatched(true);
                    }
                    updateMatched(myRelation, true);
                    updateMatched(hisRelation, true);
                    return resultVo.setMatched(true);
                }
            }
        }
        if (Constants.UserRelationType.HEARTBEAT.getType().equals(myRelation.getType())) {
            updateRelation(myRelation, relationType, false, heartBeatStatus);
            if (Constants.UserRelationType.HEARTBEAT.getType().equals(hisRelation.getType())) {
                if (Constants.YES == hisRelation.getMatched()) {
                    updateMatched(hisRelation, false);
                }
            }
            decrBeLovedIfNeeded(myRelation, heartBeatStatus);
        }
        return resultVo;
    }

    private HeartBeatResultVo handleMyExistHisNone(Long userId, Long targetUserId, Integer relationType, UserRelation myRelation, HeartBeatResultVo resultVo, int heartBeatStatus) {
        if (relationType.equals(Constants.UserRelationType.HEARTBEAT.getType())) {
            if (!myRelation.getType().equals(relationType)) {
                updateRelation(myRelation, relationType, false, heartBeatStatus);
                incrBeLoved(targetUserId, heartBeatStatus);
            }
        } else if (relationType.equals(Constants.UserRelationType.DISLIKE.getType())) {
            if (!myRelation.getType().equals(relationType)) {
                updateRelation(myRelation, relationType, false, heartBeatStatus);
                decrBeLovedIfNeeded(myRelation, heartBeatStatus);
            }
        }
        return resultVo;
    }

    private HeartBeatResultVo handleMyNoneHisExist(Long userId, Long targetUserId, Integer relationType, UserRelation hisRelation, HeartBeatResultVo resultVo, int heartBeatStatus) {
        if (relationType.equals(Constants.UserRelationType.HEARTBEAT.getType())) {
            boolean matched = Constants.UserRelationType.HEARTBEAT.getType().equals(hisRelation.getType());
            insertRelation(userId, targetUserId, relationType, matched, heartBeatStatus);
            incrBeLoved(targetUserId, heartBeatStatus);
            if (matched && Constants.NO == hisRelation.getMatched()) {
                updateMatched(hisRelation, true);
            }
            return resultVo.setMatched(matched);
        }
        insertRelation(userId, targetUserId, relationType, false, heartBeatStatus);
        return resultVo;
    }

    private HeartBeatResultVo handleBothNone(Long userId, Long targetUserId, Integer relationType, HeartBeatResultVo resultVo, int heartBeatStatus) {
        if (relationType.equals(Constants.UserRelationType.HEARTBEAT.getType())) {
            insertRelation(userId, targetUserId, relationType, false, heartBeatStatus);
            incrBeLoved(targetUserId, heartBeatStatus);
        } else if (relationType.equals(Constants.UserRelationType.DISLIKE.getType())) {
            insertRelation(userId, targetUserId, relationType, false, heartBeatStatus);
        }
        return resultVo;
    }

    private void insertRelation(Long userId, Long targetUserId, Integer type, boolean matched, int heartBeatStatus) {
        userRelationMapper.insert(new UserRelation()
                .setUserId(userId)
                .setTargetUserId(targetUserId)
                .setType(type)
                .setMatched(matched ? Constants.YES : Constants.NO)
                .setModifyTime(LocalDateTime.now())
                .setHeartBeatStatus(heartBeatStatus));
    }

    private void updateRelation(UserRelation relation, Integer type, boolean matched, int heartBeatStatus) {
        relation.setType(type).setMatched(matched ? Constants.YES : Constants.NO).setModifyTime(LocalDateTime.now()).setHeartBeatStatus(heartBeatStatus);
        userRelationMapper.updateById(relation);
    }

    private void updateMatched(UserRelation relation, boolean matched) {
        relation.setMatched(matched ? Constants.YES : Constants.NO).setModifyTime(LocalDateTime.now());
        userRelationMapper.updateById(relation);
    }

    private void incrBeLoved(Long targetUserId, Integer heartBeatStatus) {
        if (heartBeatStatus.equals(Constants.YES)) {
            counter.hincrementAndGet(RedisKeys.userBeLoveCount(), targetUserId.toString(), RedisKeys.expireTimeOneDay);
            userOtherInfoService.incrPassiveLoveCount(targetUserId, 1);
        }
    }

    private void decrBeLovedIfNeeded(UserRelation relation, Integer heartBeatStatus) {
        if (heartBeatStatus.equals(Constants.YES)) {
            if (relation.getType().equals(Constants.UserRelationType.HEARTBEAT.getType())) {
                userOtherInfoService.incrPassiveLoveCount(relation.getTargetUserId(), -1);
            }
        }
    }

    /**
     * 允许滑动限制
     * @return -1超出限制 0达到上限 1正常滑动
     */
    private Integer allowSwipe(Long userId, boolean isRealPeople) {
        String date = LocalDate.now().format(DateTimeFormatter.BASIC_ISO_DATE);
        // 查询用户可使用的奖励次数
        KeyGenerator inviteUserRewardCountKey = RedisKeys.inviteUserRewardCount(userId);
        Long inviteBrowseCount = ObjectUtils.defaultIfNull(counter.hget(inviteUserRewardCountKey, date), 0L);
        InviteConfig inviteConfig = new InviteConfig();
        String data = pair.get(Constants.PairKey.INVITE_CONFIG).of();
        if (StringUtils.isNotBlank(data)) {
            inviteConfig = JsonHelper.convert(data, InviteConfig.class);
        }
        int todayLimit = isRealPeople ? inviteConfig.getBasicBrowseCount() + inviteBrowseCount.intValue() : inviteConfig.getNoCertBrowseCount() + inviteBrowseCount.intValue();
        return swipeControlService.doSwipe(RedisKeys.swipeCountKey(userId, date).generateKey(), todayLimit, Constants.SWIPE_EXPIRE_TIME_IN_SECONDS);
    }

    @Override
    public Integer unReadCount(Long userId) {
        LambdaQueryWrapper<UserRelation> lambdaQuery = new LambdaQueryWrapper<>();
        lambdaQuery.eq(UserRelation::getTargetUserId, userId);
        lambdaQuery.eq(UserRelation::getDeleted, Constants.NOT_DELETED);
        lambdaQuery.eq(UserRelation::getType, Constants.UserRelationType.HEARTBEAT.getType());
        lambdaQuery.eq(UserRelation::getReadStatus, Constants.NO);
        lambdaQuery.eq(UserRelation::getHeartBeatStatus, Constants.YES);
        return userRelationMapper.selectCount(lambdaQuery);
    }

    @Override
    public void read(Long userId, HeartBeatReadDto heartBeatReadDto) {
        UserRelation userRelation = userRelationMapper.selectById(heartBeatReadDto.getId());
        if (Objects.isNull(userRelation)) {
            throw new IllegalStateException("心动消息不存在");
        }
        if (!userId.equals(userRelation.getTargetUserId())) {
            throw new IllegalStateException("无权限操作");
        }
        userRelation.setReadStatus(Constants.YES);
        userRelationMapper.updateById(userRelation);
    }

    @Override
    public HeartBeatUnreadCountVo countUnread(Long userId) {
        LambdaQueryWrapper<UserRelation> lambdaQuery = new LambdaQueryWrapper<>();
        lambdaQuery.eq(UserRelation::getTargetUserId, userId);
        lambdaQuery.eq(UserRelation::getType, Constants.UserRelationType.HEARTBEAT.getType());
        lambdaQuery.eq(UserRelation::getReadStatus, Constants.NO);
        lambdaQuery.eq(UserRelation::getHeartBeatStatus, Constants.YES);
        Integer heartBeatMeCount = userRelationMapper.selectCount(lambdaQuery);
        LambdaQueryWrapper<UserVisitRecord> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UserVisitRecord::getTargetUserId, userId);
        queryWrapper.eq(UserVisitRecord::getReadStatus, Constants.NO);
        Integer visitorCount = userVisitRecordMapper.selectCount(queryWrapper);
        return new HeartBeatUnreadCountVo().setHeartBeatMeCount(heartBeatMeCount).setVisitorCount(visitorCount).setTotal(heartBeatMeCount + visitorCount);
    }

    @Override
    public void clearUnread(Long userId, UserClearUnReadDto dto) {
        LambdaUpdateWrapper<UserRelation> lambdaUpdate = new LambdaUpdateWrapper<>();
        lambdaUpdate.eq(UserRelation::getUserId, userId);
        lambdaUpdate.in(UserRelation::getTargetUserId, dto.getTargetUserIds());
        lambdaUpdate.eq(UserRelation::getType, Constants.UserRelationType.HEARTBEAT.getType());
        lambdaUpdate.eq(UserRelation::getReadStatus, Constants.NO);
        lambdaUpdate.set(UserRelation::getReadStatus, Constants.YES);
        userRelationService.update(lambdaUpdate);
    }

}
