package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.contant.UserConstants;
import com.bxm.lovelink.common.dal.entity.*;
import com.bxm.lovelink.common.dal.entity.dto.matchmaker.MatchmakerRecommendQueryDto;
import com.bxm.lovelink.common.dal.entity.dto.matchmaker.RecommendMatchQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.user.UserBasicInfoVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeSimpleVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeVo;
import com.bxm.lovelink.common.dal.mapper.UserBasicInfoMapper;
import com.bxm.lovelink.common.dal.mapping.UserInfoMapping;
import com.bxm.lovelink.common.dal.service.IMeetGroupService;
import com.bxm.lovelink.common.dal.service.IUserRelationService;
import com.bxm.lovelink.common.dal.service.IUserService;
import com.bxm.lovelink.common.dal.service.MatchmakerRecommendService;
import com.bxm.lovelink.common.dal.strategy.InsertCardStrategyFactory;
import com.bxm.lovelink.common.utils.GeoDistanceUtils;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.lovelink.integration.recommend.RecommendIntegration;
import com.bxm.lovelink.rs.facade.RecommendItem;
import com.bxm.lovelink.rs.facade.RecommendRequest;
import com.bxm.lovelink.rs.facade.RecommendResponse;
import com.bxm.lovelink.rs.facade.User;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.id.IdGenerator;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.StopWatch;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


/**
 * @author weixing
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class MatchmakerRecommendServiceImpl implements MatchmakerRecommendService {
    private final UserBasicInfoMapper userBasicInfoMapper;
    private final IUserService userService;
    private final Fetcher fetcher;
    private final InsertCardStrategyFactory insertCardStrategyFactory;
    private final RecommendIntegration recommendIntegration;
    private final IdGenerator idGenerator;
    private final Updater updater;
    private final IUserRelationService userRelationService;
    private final IMeetGroupService meetGroupService;


    @Override
    public IPage<UserCompositeSimpleVo> queryRecommendList(UserComposite userComposite, MatchmakerRecommendQueryDto recommendQueryDto) {
        Page<UserCompositeSimpleVo> page = new Page<>(recommendQueryDto.getCurrent(), recommendQueryDto.getSize());
        UserBasicInfo loginBasicInfo = userComposite.getUserBasicInfo();
        if (Objects.isNull(loginBasicInfo.getGender())) {
            return getDefaultPage(page);
        }
        StopWatch stopWatch = new StopWatch();
        stopWatch.start("recommend");
        Set<String> excludeUserNumbers = new HashSet<>();
        if (StringUtils.isNotBlank(recommendQueryDto.getExcludeUserNumbers())) {
            excludeUserNumbers = new HashSet<>(Lists.newArrayList(recommendQueryDto.getExcludeUserNumbers().split(",")));
        }
        RecommendRequest recommendRequest = buildRecommendRequest(loginBasicInfo, userComposite.getUserLoveCondition(), excludeUserNumbers);
        RecommendResponse recommendResponse = recommendIntegration.recommend(recommendRequest);
        stopWatch.stop();
        if (Objects.isNull(recommendResponse) || CollectionUtils.isEmpty(recommendResponse.getRecommendList())) {
            log.warn("queryRecommendList error recommendResponse is null, recommendRequest:{}", JsonHelper.convert(recommendRequest));
            return getDefaultPage(page);
        }
        List<RecommendItem> recommendList = recommendResponse.getRecommendList();
        recommendList = recommendList.subList(0, Math.min(recommendQueryDto.getSize(), recommendList.size()));
        if (CollectionUtils.isEmpty(recommendList)) {
            return getDefaultPage(page);
        }
        List<Long> userIdList = recommendList.stream()
                .filter(Objects::nonNull).map(RecommendItem::getId)
                .filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userIdList)) {
            return getDefaultPage(page);
        }
        stopWatch.start("getUserCompositeList");
        List<UserComposite> userCompositeList = getUserCompositeList(userIdList);
        stopWatch.stop();
        if (CollectionUtils.isEmpty(userCompositeList)) {
            return getDefaultPage(page);
        }
        // 按照userId顺序重排
        userCompositeList = reorderUserCompositeList(userCompositeList, userIdList);
        // 插入被分享用户
        stopWatch.start("insertSharedUserCard");
        insertSharedUserCard(loginBasicInfo, userCompositeList);
        stopWatch.stop();
        stopWatch.start("getUserCompositeSimpleVos");
        List<UserCompositeSimpleVo> simpleVoList = getUserCompositeSimpleVos(userCompositeList, loginBasicInfo);
        stopWatch.stop();
        // 处理插卡逻辑
        stopWatch.start("handleInsertCard");
        handleInsertCard(loginBasicInfo, userComposite.getUserOtherInfo(), recommendQueryDto, simpleVoList);
        stopWatch.stop();
        page.setRecords(simpleVoList);
        log.info("queryRecommendList stopWatch:{}", stopWatch.prettyPrint());
        return page;
    }

    @Override
    public List<UserCompositeSimpleVo> getMatches(RecommendMatchQueryDto dto) {
        UserComposite userComposite = userService.getCompositeById(dto.getUserId());
        if (Objects.isNull(userComposite) || Objects.isNull(userComposite.getUserBasicInfo())) {
            return Collections.emptyList();
        }
        Set<String> excludeUserNumbers = new HashSet<>();
        if (CollectionUtils.isNotEmpty(dto.getExcludeUserNumbers())) {
            excludeUserNumbers = new HashSet<>(dto.getExcludeUserNumbers());
        }
        RecommendRequest recommendRequest = buildRecommendRequest(userComposite.getUserBasicInfo(), userComposite.getUserLoveCondition(), excludeUserNumbers);
        RecommendResponse recommendResponse = recommendIntegration.recommend(recommendRequest);
        if (Objects.isNull(recommendResponse) || CollectionUtils.isEmpty(recommendResponse.getRecommendList())) {
            return Collections.emptyList();
        }
        List<RecommendItem> recommendList = recommendResponse.getRecommendList();
        recommendList = recommendList.subList(0, Math.min(dto.getCount(), recommendList.size()));
        if (CollectionUtils.isEmpty(recommendList)) {
            return Collections.emptyList();
        }
        List<Long> userIdList = recommendList.stream()
                .filter(Objects::nonNull).map(RecommendItem::getId)
                .filter(Objects::nonNull).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.emptyList();
        }
        List<UserComposite> userCompositeList = getUserCompositeList(userIdList);
        if (CollectionUtils.isEmpty(userCompositeList)) {
            return Collections.emptyList();
        }
        userCompositeList = reorderUserCompositeList(userCompositeList, userIdList);
        return getUserCompositeSimpleVos(userCompositeList, userComposite.getUserBasicInfo());
    }

    private List<UserComposite> reorderUserCompositeList(List<UserComposite> userCompositeList, List<Long> userIdList) {
        Map<Long, UserComposite> userMap = userCompositeList.stream()
                .filter(Objects::nonNull)
                .filter(u -> u.getUser() != null && u.getUser().getId() != null)
                .collect(Collectors.toMap(
                        u -> u.getUser().getId(),
                        Function.identity(),
                        (a, b) -> a
                ));

        return userIdList.stream()
                .map(userMap::get)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private List<UserComposite> getUserCompositeList(List<Long> userIdList) {
        try {
            return userService.getCompositePartByUserIdsFromCache(userIdList);
        } catch (Exception e) {
            log.warn("从缓存获取用户信息失败，降级查询数据库，userIds={}", userIdList, e);
            return userBasicInfoMapper.selectByIds(userIdList);
        }
    }

    private static Page<UserCompositeSimpleVo> getDefaultPage(Page<UserCompositeSimpleVo> page) {
        return page.setRecords(Lists.newArrayList(new UserCompositeSimpleVo().setCardType(Constants.CardSubType.INVITE).setSwipeEffect(Constants.SwipeEffect.FIXED)));
    }

    private List<UserCompositeSimpleVo> getUserCompositeSimpleVos(List<UserComposite> userCompositeList, UserBasicInfo loginBasicInfo) {
        List<UserCompositeVo> convert = userService.convert(userCompositeList);
        convert.forEach(vo -> {
            if (Objects.nonNull(vo)) {
                UserBasicInfoVo basicInfo = vo.getUserBasicInfo();
                Double distance = GeoDistanceUtils.calculateDistance(loginBasicInfo.getLatitude(), loginBasicInfo.getLongitude(), basicInfo.getLatitude(), basicInfo.getLongitude());
                basicInfo.setDistance(distance);
                basicInfo.setSchoolName(Objects.equals(basicInfo.getSchoolHide(), Constants.NO) ? null : basicInfo.getSchoolName());
            }
        });
        return UserInfoMapping.INSTANCE.toList(convert);
    }

    private RecommendRequest buildRecommendRequest(UserBasicInfo userBasicInfo, UserLoveCondition userLoveCondition, Set<String> excludeUserNumbers) {
        RecommendRequest recommendRequest = new RecommendRequest();
        recommendRequest.setRequestId(idGenerator.next());
        recommendRequest.setContainUserInfo(true);
        User user = UserInfoMapping.INSTANCE.toRecommendUser(userBasicInfo);
        if (Objects.nonNull(userLoveCondition)) {
            user.setAgeRangeMax(userLoveCondition.getAgeRangeMax());
            user.setAgeRangeMin(userLoveCondition.getAgeRangeMin());
            user.setHeightRangeMin(userLoveCondition.getHeightRangeMin());
            user.setHeightRangeMax(userLoveCondition.getHeightRangeMax());
            if (!CollectionUtils.isEmpty(userLoveCondition.getEduRange())) {
                user.setEduRange(userLoveCondition.getEduRange());
            }
            if (!CollectionUtils.isEmpty(userLoveCondition.getYearRevenueRange())) {
                user.setYearRevenueRange(userLoveCondition.getYearRevenueRange());
            }
        }
        recommendRequest.setUser(user);
        if (CollectionUtils.isNotEmpty(excludeUserNumbers)) {
            recommendRequest.setExcludeUserNumbers(excludeUserNumbers);
        }
        List<UserRelation> userRelations = userRelationService.queryAll(user.getId());
        if (CollectionUtils.isNotEmpty(userRelations)) {
            List<Long> likeUserList = Lists.newArrayList();
            List<Long> dislikeUserList = Lists.newArrayList();
            for (UserRelation userRelation : userRelations) {
                if (Objects.equals(Constants.UserRelationType.HEARTBEAT.getType(), userRelation.getType())) {
                    likeUserList.add(userRelation.getTargetUserId());
                } else if ((Objects.equals(Constants.UserRelationType.DISLIKE.getType(), userRelation.getType()))) {
                    dislikeUserList.add(userRelation.getTargetUserId());
                }
            }
            // 设置用户心动的用户
            user.setLikeUserList(likeUserList);
            // 设置用户不喜欢的用户
            user.setDislikeUserList(dislikeUserList);
        }

        // 设置用户主动邀约见面的用户
        List<MeetGroup> meetGroups = meetGroupService.queryAll(user.getId());
        if (CollectionUtils.isNotEmpty(meetGroups)) {
            user.setInviteUserList(meetGroups.stream().map(MeetGroup::getAcceptUserId).collect(Collectors.toList()));
        }

        return recommendRequest;
    }

    private void handleInsertCard(UserBasicInfo loginBasicInfo, UserOtherInfo userOtherInfo, MatchmakerRecommendQueryDto recommendQueryDto, List<UserCompositeSimpleVo> simpleVoList) {
        // 处理插卡
        String date = LocalDate.now().format(DateTimeFormatter.BASIC_ISO_DATE);
        // 获取当前用户今天滑动次数
        Integer todaySwipeCount = Optional.ofNullable(fetcher.fetch(RedisKeys.swipeCountKey(loginBasicInfo.getUserId(), date), Integer.class)).orElse(0);
        InsertCardContext context = new InsertCardContext();
        context.setUserId(loginBasicInfo.getUserId());
        context.setTodaySwipeCount(todaySwipeCount);
        context.setRequestSize(simpleVoList.size());
        context.setUserInfoComplete(userOtherInfo.getBasicInfoGuideFinished().equals(Constants.YES));
        context.setRealName(!CollectionUtils.isEmpty(loginBasicInfo.getCertStatus()) && loginBasicInfo.getCertStatus().contains(UserConstants.CertTypeEnum.REAL_PEOPLE.getCode()));
        if (StringUtils.isNotBlank(recommendQueryDto.getExcludeUserNumbers())) {
            context.setWaitSwipeCount(new HashSet<>(Lists.newArrayList(recommendQueryDto.getExcludeUserNumbers().split(","))).size());
        }
        List<InsertCard> insertCards = insertCardStrategyFactory.computeAllInsertCards(context);
        if (insertCards.isEmpty()) {
            return;
        }
        Set<Integer> insertedPositions = new HashSet<>();
        for (InsertCard card : insertCards) {
            int index = card.getIndex();
            Integer cardSubType = card.getCardSubType();
            if (index < 0 || index > simpleVoList.size()) {
                continue;
            }
            if (cardSubType == Constants.CardSubType.REAL_NAME) {
                if (index < simpleVoList.size()) {
                    simpleVoList.get(index)
                            .setPopType(cardSubType)
                            .setSwipeEffect(Constants.SwipeEffect.REBOUNCE);
                }
            } else {
                if (insertedPositions.add(index)) {
                    simpleVoList.add(index,
                            new UserCompositeSimpleVo()
                                    .setCardType(cardSubType)
                                    .setSwipeEffect(card.getSwipeEffect()));
                }
            }
        }
        // 截取分享卡片及之前的卡片
        for (int i = 0; i < simpleVoList.size(); i++) {
            UserCompositeSimpleVo vo = simpleVoList.get(i);
            if (vo.getCardType() != null && vo.getCardType() == Constants.CardSubType.INVITE) {
                simpleVoList.subList(i + 1, simpleVoList.size()).clear();
                break;
            }
        }
    }

    private void insertSharedUserCard(UserBasicInfo loginBasicInfo, List<UserComposite> userComposites) {
        // 被邀请用户是否有被分享用户
        String userId = fetcher.fetch(RedisKeys.userFirstAppointDetailPage(loginBasicInfo.getUserId()), String.class);
        if (StringUtils.isBlank(userId)) {
            return;
        }
        // 查询用户信息 插入第一张
        List<UserComposite> userCompositeList = getUserCompositeList(Collections.singletonList(Long.valueOf(userId)));
        if (CollectionUtils.isEmpty(userCompositeList)) {
            updater.remove(RedisKeys.userFirstAppointDetailPage(loginBasicInfo.getUserId()));
            return;
        }
        UserComposite userComposite = userCompositeList.iterator().next();
        boolean shouldInsert = Optional.ofNullable(userComposite.getUserBasicInfo())
                .map(info -> !info.getGender().equals(loginBasicInfo.getGender()))
                .orElse(false);
        if (shouldInsert) {
            userComposites.removeIf(vo -> userComposite.getUser().getId().equals(vo.getUser().getId()));
            userComposites.add(0, userComposite);
        }
        // 清除缓存
        updater.remove(RedisKeys.userFirstAppointDetailPage(loginBasicInfo.getUserId()));
    }
}
