package com.bxm.lovelink.common.dal.service.impl;


import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.service.ISwipeControlService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.Collections;
import java.util.Objects;

/**
 * 滑动控制服务实现类
 * @author: weixing
 */
@Slf4j
@Service
public class SwipeControlServiceImpl implements ISwipeControlService, InitializingBean {
    private final JedisPool jedisPool;
    private String scriptSha;

    public SwipeControlServiceImpl(JedisPool jedisPool) {
        this.jedisPool = jedisPool;
    }

    /**
     * 滑动控制逻辑
     * @return -1超出限制 0达到上限 1正常滑动
     */
    @Override
    public Integer doSwipe(String redisKey, Integer todayLimit, Long expireTime)  {
        if (StringUtils.isBlank(redisKey) || Objects.isNull(todayLimit) || Objects.isNull(expireTime)) {
            log.error("滑动控制参数异常 redisKey={},todayLimit={},expireTime={}", redisKey, todayLimit, expireTime);
            return Constants.SwipeLimitStatus.NORMAL;
        }
        try (Jedis resource = jedisPool.getResource()) {
            Long result = (Long) resource.evalsha(scriptSha,
                    Collections.singletonList(redisKey),
                    Lists.newArrayList(String.valueOf(todayLimit), String.valueOf(expireTime)));
            return result.intValue();
        }
    }

    @Override
    public void afterPropertiesSet(){
        String script =
                "local key = KEYS[1]\n" +
                        "local todayLimit = tonumber(ARGV[1])\n" +
                        "local expireTime = tonumber(ARGV[2])\n" +
                        "local current = redis.call('INCR', key)\n" +
                        "if current == 1 then\n" +
                        "    redis.call('EXPIRE', key, expireTime)\n" +
                        "end\n" +
                        "if current > todayLimit then\n" +
                        "    return -1\n" +
                        "elseif current == todayLimit then\n" +
                        "    return 0\n" +
                        "else\n" +
                        "    return 1\n" +
                        "end";
        try (Jedis resource = jedisPool.getResource()) {
            scriptSha = resource.scriptLoad(script);
        }
    }
}
