package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.UserAction;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserComposite;
import com.bxm.lovelink.common.dal.entity.UserRuleTag;
import com.bxm.lovelink.common.dal.entity.dto.user.UserBasicInfoDto;
import com.bxm.lovelink.common.dal.entity.dto.user.UserQueryDto;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeVo;
import com.bxm.lovelink.common.dal.mapper.UserBasicInfoMapper;
import com.bxm.lovelink.common.dal.mapping.UserInfoMapping;
import com.bxm.lovelink.common.dal.service.*;
import com.bxm.lovelink.common.dal.user.basicinfo.UserBasicInfoBuilderChain;
import com.bxm.lovelink.common.dal.user.basicinfo.UserBasicInfoBuilderContext;
import com.bxm.lovelink.common.enums.UserBasicInfoCompletionEnum;
import com.bxm.lovelink.common.event.promotion.UserActionEvent;
import com.bxm.lovelink.common.event.user.UserBasicInfoUpdateEvent;
import com.bxm.lovelink.common.event.user.UserCompleteStatusEvent;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.warcar.integration.eventbus.EventPark;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * <p>
 * 用户的基本信息 服务实现类
 * </p>
 *
 * @author zhangdong
 * @since 2025-05-15
 */
@Service
@Slf4j
public class UserBasicInfoServiceImpl extends ServiceImpl<UserBasicInfoMapper, UserBasicInfo> implements IUserBasicInfoService {


    private final IUserCertService userCertService;
    private final UserBasicInfoBuilderChain userBasicInfoBuilderChain;
    private final IUserService userService;
    private final RedissonClient redissonClient;
    private final IUserRuleTagService userRuleTagService;
    private final IUserLoveConditionService userLoveConditionService;
    private final EventPark eventPark;


    public UserBasicInfoServiceImpl(@Lazy IUserCertService userCertService, UserBasicInfoBuilderChain userBasicInfoBuilderChain, @Lazy IUserService userService, RedissonClient redissonClient, IUserRuleTagService userRuleTagService, @Lazy IUserLoveConditionService userLoveConditionService, EventPark eventPark) {
        this.userCertService = userCertService;
        this.userBasicInfoBuilderChain = userBasicInfoBuilderChain;
        this.userService = userService;
        this.redissonClient = redissonClient;
        this.userRuleTagService = userRuleTagService;
        this.userLoveConditionService = userLoveConditionService;
        this.eventPark = eventPark;
    }

    @Override
    public UserBasicInfo getOneByUserId(Long userId) {
        return getOne(new LambdaQueryWrapper<UserBasicInfo>().eq(UserBasicInfo::getUserId, userId));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByDto(UserBasicInfoDto dto, UserComposite composite) {
        UserBasicInfo existingBasicInfo = composite.getUserBasicInfo();
        Long userId = existingBasicInfo.getUserId();
        RLock lock = redissonClient.getLock(RedisKeys.userBasicInfoUpdateLock(userId).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            throw new BusinessException("信息更新中，请稍后再试");
        }
        try {
            UserBasicInfo basicInfo = UserInfoMapping.INSTANCE.dtoToDo(dto);
            basicInfo.setId(existingBasicInfo.getId()).setUserId(userId);
            UserBasicInfoBuilderContext context = new UserBasicInfoBuilderContext()
                    .setUserComposite(composite)
                    .setExistBasicInfo(existingBasicInfo)
                    .setBasicInfoDto(dto);
            userBasicInfoBuilderChain.buildAndCheck(context, basicInfo);
            updateAndCalculateById(basicInfo);
        } finally {
            lock.unlock();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean updateAndCalculateById(UserBasicInfo entity) {
        boolean result = super.updateById(entity);
        if (!result) {
            return false;
        }
        UserBasicInfo userBasicInfo = getById(entity.getId());
        UserBasicInfo updateBasicInfo = new UserBasicInfo().setId(userBasicInfo.getId());
        //计算信息完成度
        Integer completion = UserBasicInfoCompletionEnum.calculateCompletion(userBasicInfo);
        updateBasicInfo.setInfoCompleteProgress(completion);
        //计算状态标签
        UserComposite userComposite = new UserComposite().setUserBasicInfo(userBasicInfo)
                .setUserCerts(userCertService.getListNewByUserId(userBasicInfo.getUserId()));
        Map<Integer, UserRuleTag> map = userRuleTagService.calculate(userComposite);
        List<UserRuleTag> displayTag = new ArrayList<>();
        List<UserRuleTag> hideTag = new ArrayList<>();
        List<Integer> hideTypes = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(userBasicInfo.getHideTag())) {
            hideTypes = userBasicInfo.getHideTag().stream().map(UserRuleTag::getType).collect(Collectors.toList());
        }
        for (Map.Entry<Integer, UserRuleTag> entry : map.entrySet()) {
            if (hideTypes.contains(entry.getKey())) {
                hideTag.add(entry.getValue());
            } else {
                displayTag.add(entry.getValue());
            }
        }
        updateBasicInfo.setRuleTag(displayTag);
        updateBasicInfo.setHideTag(hideTag);
        //最后执行更新
        super.updateById(updateBasicInfo);
        //事务执行完成以后 再发送事件
        TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronization() {
            @Override
            public void afterCommit() {
                // 发送注册完成事件
                eventPark.post(new UserActionEvent(this, new UserAction().setUserId(entity.getUserId()).setType(UserAction.REGISTER)));

                if (completion >= Constants.INFO_COMPLETION_THRESHOLD) {
                    // 发送资料完善推广事件
                    eventPark.post(new UserActionEvent(this, new UserAction().setUserId(entity.getUserId()).setType(UserAction.INFO_COMPLETE)));
                }
                //发送更新事件
                eventPark.post(new UserBasicInfoUpdateEvent(this, userBasicInfo.getUserId(), userBasicInfo));
                eventPark.post(new UserCompleteStatusEvent(this, userBasicInfo.getUserId()));
            }
        });
        return true;
    }

    @Override
    public void updateVipExpireTime(Long userId, int validDays) {
        if (validDays <= 0) {
            log.error("illegal validDays {}", validDays);
            return;
        }
        UserBasicInfo basicInfo = getOneByUserId(userId);
        LocalDateTime vipExpireTime = computeVipExpireTime(userId, validDays);

        UserBasicInfo userBasicInfo = new UserBasicInfo();
        userBasicInfo.setId(basicInfo.getId());
        userBasicInfo.setVipExpireTime(vipExpireTime);
        if (!this.updateById(userBasicInfo)) {
            throw new IllegalStateException("更新用户vip到期时间失败");
        }
    }

    @Override
    public LocalDateTime computeVipExpireTime(Long userId, int validDays) {
        UserBasicInfo basicInfo = getOneByUserId(userId);
        LocalDateTime vipExpireTime = basicInfo.getVipExpireTime();
        LocalDateTime now = LocalDateTime.now();
        if (vipExpireTime == null || vipExpireTime.isBefore(LocalDateTime.now())) {
            vipExpireTime = now.plusDays(validDays);
        } else {
            // 续期
            vipExpireTime = vipExpireTime.plusDays(validDays);
        }
        return vipExpireTime;
    }

    @Override
    public IPage<UserCompositeVo> list(UserQueryDto dto) {
        LocalDate now = LocalDate.now();
        if (dto.getAgeMin() != null) {
            dto.setBirthdayEnd(now.minusYears(dto.getAgeMin()));
        }
        if (dto.getAgeMax() != null) {
            dto.setBirthdayStart(now.minusYears(dto.getAgeMax()));
        }
        Page<UserComposite> compositePage = baseMapper.queryList(new Page<>(dto.getCurrent(), dto.getSize()), dto);
        Page<UserCompositeVo> voPage = UserInfoMapping.INSTANCE.toUserPage(compositePage);
        voPage.setRecords(userService.convert(compositePage.getRecords()));
        return voPage;
    }

    @Override
    public List<UserBasicInfo> listByUserIds(List<Long> userIds) {
        if (CollectionUtils.isEmpty(userIds)) {
            return Collections.emptyList();
        }
        return baseMapper.selectList(new LambdaQueryWrapper<UserBasicInfo>().in(UserBasicInfo::getUserId, userIds));
    }

    @Override
    public List<UserBasicInfo> listIncludeDeleted(List<Long> userIds) {
        return baseMapper.listIncludeDeleted(userIds);
    }
}
