package com.bxm.lovelink.common.dal.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.dal.entity.*;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.lovelink.common.contant.UserConstants;
import com.bxm.lovelink.common.dal.mapper.UserCertMapper;
import com.bxm.lovelink.common.dal.service.IUserBasicInfoService;
import com.bxm.lovelink.common.dal.service.IUserCertService;
import com.bxm.lovelink.common.event.promotion.UserActionEvent;
import com.bxm.lovelink.common.event.user.UserCompleteStatusEvent;
import com.bxm.lovelink.integration.identify.IdentifyIntegration;
import com.bxm.lovelink.integration.identify.IdentifyResult;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * <p>
 * 用户资料认证 服务实现类
 * </p>
 *
 * @author zhangdong
 * @since 2025-05-15
 */
@Service
@Slf4j
public class UserCertServiceImpl extends ServiceImpl<UserCertMapper, UserCert> implements IUserCertService {

    private final IUserBasicInfoService userBasicInfoService;
    private final RedissonClient redissonClient;
    private final IdentifyIntegration identifyIntegration;
    private final EventPark eventPark;


    public UserCertServiceImpl(IUserBasicInfoService userBasicInfoService, RedissonClient redissonClient, IdentifyIntegration identifyIntegration, EventPark eventPark) {
        this.userBasicInfoService = userBasicInfoService;
        this.redissonClient = redissonClient;
        this.identifyIntegration = identifyIntegration;
        this.eventPark = eventPark;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cert(UserCert userCert) {
        if (userCert == null || userCert.getUserId() == null || userCert.getCertType() == null) {
            throw new IllegalArgumentException("认证信息不能为空");
        }
        RLock lock = redissonClient.getLock(RedisKeys.userCertLock(userCert.getUserId(), userCert.getCertType()).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            throw new IllegalArgumentException("认证中，请稍后再试");
        }

        try {
            check(userCert.getUserId(), userCert.getCertType());
            save(userCert);
            updateCertProgress(userCert.getUserId());
        } finally {
            lock.unlock();
        }
    }

    /**
     * 更新认证结果
     * 调用前必须已经创建认证中的记录
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateCertByApi(UserCert userCert) {
        if (userCert == null) {
            return;
        }
        if (!userCert.isByApi() || !userCert.isWaiting()) {
            return;
        }

        RLock lock = redissonClient.getLock(RedisKeys.userCertLock(userCert.getUserId(), userCert.getCertType()).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            throw new IllegalArgumentException("认证中，请稍后再试");
        }

        try {
            UserCertPeople certPeople = JsonHelper.convert(userCert.getCertContent(), UserCertPeople.class);
            IdentifyResult identifyResult = identifyIntegration.queryResult(certPeople.getName(), certPeople.getCardNumber(), userCert.getCertType());
            userCert.setCertResult(identifyResult.getCertResult());
            userCert.setCertStatus(identifyResult.isSuccess() ? UserCert.STATUS_PASS : UserCert.STATUS_FAIL);

            updateById(userCert);
            updateCertProgress(userCert.getUserId());

            if (identifyResult.isSuccess()) {
                if (userCert.getCertType().equals(UserConstants.CertTypeEnum.EDU.getCode())) {
                    eventPark.post(new UserActionEvent(this, new UserAction().setUserId(userCert.getUserId()).setType(UserAction.EDUCATION).setEduHighest(userCert.getEduHighest())));
                }
                if (userCert.getCertType().equals(UserConstants.CertTypeEnum.MARRIED.getCode()) && userCert.isUnMarried()) {
                    eventPark.post(new UserActionEvent(this, new UserAction().setUserId(userCert.getUserId()).setType(UserAction.MARRIAGE)));
                }
            }
        } finally {
            lock.unlock();
        }
    }

    private void updateCertProgress(Long userId) {
        List<UserCert> userCerts = getListNewByUserId(userId);
        int size = UserConstants.CertTypeEnum.getNeededCertTypes().size();
        int count = 0;
        List<Integer> certStatus = new ArrayList<>();
        for (UserCert value : userCerts) {
            if (value.isSuccess()) {
                count++;
            }
            // 这判断了车辆认证有车 前端才显示图标
            if (value.isSuccessWithCarNum()) {
                certStatus.add(value.getCertType());
            }
        }
        int percent;
        if (size == count) {
            percent = 100;
        } else {
            percent = 100 / size * count;
        }

        UserBasicInfo basicInfo = userBasicInfoService.getOneByUserId(userId);
        Integer trustPercent = computeTrustPercent(userCerts, basicInfo);

        userBasicInfoService.updateById(new UserBasicInfo().setId(basicInfo.getId()).setInfoCertProgress(percent)
                .setTrustPercent(trustPercent)
                .setCertStatus(certStatus));
        eventPark.post(new UserCompleteStatusEvent(this, userId));
    }

    @Override
    public List<UserCert> getListByUserId(Long userId) {
        return list(new LambdaQueryWrapper<UserCert>().eq(UserCert::getUserId, userId).orderByDesc(UserCert::getId));
    }

    @Override
    public List<UserCert> getListNewByUserId(Long userId) {
        Map<Integer, UserCert> map = list(new LambdaQueryWrapper<UserCert>()
                .eq(UserCert::getUserId, userId)
                .orderByDesc(UserCert::getId)).stream()
                .collect(Collectors.groupingBy(UserCert::getCertType
                        , Collectors.collectingAndThen(Collectors.toList(), x -> x.isEmpty() ? null : x.get(0))));
        return new ArrayList<>(map.values());
    }

    @Override
    public UserCert getOneByUserIdAndType(Long userId, Integer type) {
        return getOne(new LambdaQueryWrapper<UserCert>().eq(UserCert::getUserId, userId)
                .eq(UserCert::getCertType, type)
                .orderByDesc(UserCert::getId).last("LIMIT 1"));
    }

    @Override
    public void check(Long userId, Integer type) {
        UserCert userCert = getOneByUserIdAndType(userId, type);
        if (userCert == null) {
            return;
        }
        if (userCert.isWaiting()) {
            throw new IllegalArgumentException("认证中，请等待认证结果");
        }
        if (userCert.isSuccess() && userCert.isRealName()) {
            throw new IllegalArgumentException("认证已经成功，请勿重复认证");
        }
    }

    private Integer computeTrustPercent(List<UserCert> userCerts, UserBasicInfo basicInfo) {
        // 声明无车不参与信任度计算
        List<Integer> neededCertTypes = UserConstants.CertTypeEnum.getNeededCertTypes();
        if (basicInfo.declareNoCar()) {
            neededCertTypes.remove(UserConstants.CertTypeEnum.CAR.getCode());
            userCerts.removeIf(c -> UserConstants.CertTypeEnum.CAR.getCode().equals(c.getCertType()));
        }

        int count = 0;
        for (UserCert value : userCerts) {
            if (value.isSuccessWithCarNum()) {
                count++;
            }
        }
        int size = neededCertTypes.size();
        int trustPercent;
        if (size == count) {
            trustPercent = 99;
        } else {
            trustPercent = ((count * 99) + (size - count) * 50) / size;
        }
        return trustPercent;
    }
}
