package com.bxm.lovelink.common.dal.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.UserMeetTicket;
import com.bxm.lovelink.common.dal.entity.UserMeetTicketRecord;
import com.bxm.lovelink.common.dal.entity.UserOrder;
import com.bxm.lovelink.common.dal.entity.UserTicketRecordExt;
import com.bxm.lovelink.common.dal.entity.dto.meetticket.ReturnTicketOperateDto;
import com.bxm.lovelink.common.dal.entity.dto.meetticket.TicketFinishGiveOperateDto;
import com.bxm.lovelink.common.dal.entity.dto.meetticket.TicketOperateDto;
import com.bxm.lovelink.common.dal.mapper.UserMeetTicketMapper;
import com.bxm.lovelink.common.dal.service.IUserMeetTicketRecordService;
import com.bxm.lovelink.common.dal.service.IUserMeetTicketService;
import com.bxm.lovelink.common.dal.service.IUserOrderService;
import com.bxm.lovelink.common.dal.service.IUserOtherInfoService;
import com.bxm.lovelink.common.enums.TicketRecordTypeEnum;
import com.bxm.lovelink.common.enums.TicketReturnTypeEnum;
import com.bxm.lovelink.common.exception.BusinessException;
import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * <p>
 * 用户红豆 服务实现类
 * </p>
 *
 * @author tangxiao
 * @since 2025-05-15
 */
@Service
public class UserMeetTicketServiceImpl extends ServiceImpl<UserMeetTicketMapper, UserMeetTicket> implements IUserMeetTicketService {

    private final IUserMeetTicketRecordService ticketRecordService;
    private IUserOrderService userOrderService;
    private final IUserOtherInfoService otherInfoService;

    public UserMeetTicketServiceImpl(IUserMeetTicketRecordService ticketRecordService, IUserOtherInfoService otherInfoService) {
        this.ticketRecordService = ticketRecordService;
        this.otherInfoService = otherInfoService;
    }

    // 为了解决循环依赖
    @Autowired
    public void setUserOrderService(IUserOrderService userOrderService) {
        this.userOrderService = userOrderService;
    }

    @Override
    public void rechargeTicket(TicketOperateDto dto) {
        Preconditions.checkArgument(dto.getType() != null, "缺少type参数");

        Long orderId = dto.getConnectId();
        UserOrder userOrder = userOrderService.getById(orderId);
        if (Objects.isNull(userOrder)) {
            throw new IllegalStateException("订单不存在");
        }
        if (userOrder.getStatus() != Constants.UserOrder.STATUS_SUCCESS) {
            throw new IllegalStateException("订单未支付成功");
        }

        save(new UserMeetTicket()
                .setUserId(dto.getUserId())
                .setMeetTicketProductId(userOrder.getMeetTicketProductId())
                .setMeetTicketProductName(userOrder.getProductName())
                .setTotalAmount(dto.getAmount())
                .setRemainAmount(dto.getAmount())
                .setExpireTime(LocalDateTime.now().plusDays(Constants.UserMeetTicket.TICKET_VALID_DAYS))
                .setTicketId(UUID.randomUUID().toString())
                .setPrice(userOrder.getPaymentAmount())
                .setStatus(UserMeetTicket.STATUS_AVAILABLE));

        JSONObject productExtObj = JSONObject.parseObject(userOrder.getProductExtJson());

        UserTicketRecordExt userTicketRecordExt = new UserTicketRecordExt()
                .setTitle(userOrder.getProductName())
                .setDetailTitle(userOrder.getProductName())
                .setDescription(userOrder.getProductName())
                .setVipExpireTime(productExtObj.getString(Constants.UserOrder.ExtJsonKey.VIP_EXPIRE_TIME))
                .setTicketExpireTime(LocalDateTime.now().plusDays(Constants.UserMeetTicket.TICKET_VALID_DAYS))
                .setInviteTime(LocalDateTime.now())
                .setRemark(String.format("红豆有效期%s天", Constants.UserMeetTicket.TICKET_VALID_DAYS));

        saveRecord(dto.getUserId(), dto.getAmount(), dto.getConnectId(), userTicketRecordExt, dto.getType());
    }

    @Override
    @Transactional(rollbackFor = Exception.class, timeout = 20)
    public void giveTicket(TicketFinishGiveOperateDto dto) {
        TicketRecordTypeEnum type = dto.getType();
        Preconditions.checkArgument(type != null, "缺少type参数");


        LocalDateTime expireTime = LocalDateTime.now().plusDays(Constants.UserMeetTicket.TICKET_VALID_DAYS);
        save(new UserMeetTicket()
                .setUserId(dto.getUserId())
                .setTotalAmount(dto.getAmount())
                .setRemainAmount(dto.getAmount())
                .setExpireTime(expireTime)
                .setTicketId(UUID.randomUUID().toString())
                .setPrice(BigDecimal.ZERO)
                .setStatus(UserMeetTicket.STATUS_AVAILABLE));

        UserTicketRecordExt userTicketRecordExt = new UserTicketRecordExt()
                .setTitle("平台赠送")
                .setDetailTitle(String.format("平台赠送%s颗红豆", dto.getAmount()))
                .setDescription(String.format("%s 赠送%s颗红豆", type.getDesc(), dto.getAmount()))
                .setTicketExpireTime(expireTime)
                .setInviteTime(LocalDateTime.now())
                .setRemark(String.format("红豆有效期%s天", Constants.UserMeetTicket.TICKET_VALID_DAYS));

        saveRecord(dto.getUserId(), dto.getAmount(), dto.getConnectId(), userTicketRecordExt, type);
        //记录是否已经赠送
        otherInfoService.updateInfoFinishGiveStage(dto.getUserId(), dto.getStage());
    }

    @Override
    @Transactional(rollbackFor = Exception.class, timeout = 20)
    public List<Long> useTicket(TicketOperateDto dto) {
        int remainAmount = getUserRemainAmount(dto.getUserId());
        if (remainAmount < dto.getAmount()) {
            throw new IllegalStateException("没有足够的红豆");
        }

        minus(dto.getUserId(), dto.getAmount());

        UserTicketRecordExt ext = new UserTicketRecordExt()
                .setTitle(dto.getRemark() + " 支出")
                .setDescription(dto.getRemark() + String.format(" 支付%s颗红豆", dto.getAmount()))
                .setInviteContent(dto.getRemark())
                .setRemark(String.format("超过%s小时未确认，红豆将自动退回", Constants.MeetGroupConst.INVITE_EXPIRE_TIME_HOUR))
                .setInviteTime(LocalDateTime.now());

        saveRecord(dto.getUserId(), dto.getAmount(), dto.getConnectId(), ext, TicketRecordTypeEnum.USE);
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class, timeout = 10)
    public void returnTicket(ReturnTicketOperateDto dto) {
        Integer amount = dto.getAmount();
        if (TicketReturnTypeEnum.REFUSE.equals(dto.getReturnType())) {
            // 邀约被对方拒绝，退还80%的红豆
            amount = (int) Math.ceil(amount * Constants.UserMeetTicket.RETURN_RATE);
        }

        plus(dto.getUserId(), amount);

        // 查找关联的支出信息
        UserMeetTicketRecord relateUseInfo = ticketRecordService.getOne(new LambdaQueryWrapper<UserMeetTicketRecord>()
                .eq(UserMeetTicketRecord::getUserId, dto.getUserId())
                .eq(UserMeetTicketRecord::getType, TicketRecordTypeEnum.USE.getOrdinal())
                .eq(UserMeetTicketRecord::getConnectId, dto.getConnectId()));
        if (Objects.isNull(relateUseInfo)) {
            throw new IllegalStateException("退红豆时无法找到关联的支出信息 - " + dto.getConnectId());
        }
        UserTicketRecordExt relateInfoExt = relateUseInfo.getExtJson();
        String inviteInfo = relateInfoExt.getInviteContent();
        UserTicketRecordExt userTicketRecordExt = new UserTicketRecordExt()
                .setTitle(inviteInfo + " " + dto.getReturnType().getDesc() + " 退红豆")
                .setDetailTitle(String.format("退%s颗红豆", amount))
                .setDescription(relateInfoExt.getDescription())
                .setReturnReason(dto.getReturnType().getDesc())
                .setInviteTime(relateInfoExt.getInviteTime())
                .setReturnTime(LocalDateTime.now());

        saveRecord(dto.getUserId(), amount, dto.getConnectId(), userTicketRecordExt, TicketRecordTypeEnum.RETURN);
    }

    @Override
    public List<UserMeetTicket> listAvailableTicket(Long userId) {
        return list(new LambdaQueryWrapper<UserMeetTicket>()
                .eq(UserMeetTicket::getUserId, userId)
                .eq(UserMeetTicket::getStatus, UserMeetTicket.STATUS_AVAILABLE)
                .gt(UserMeetTicket::getExpireTime, LocalDateTime.now())
        );
    }

    public int getUserRemainAmount(Long userId) {
        List<UserMeetTicket> list = this.list(new LambdaQueryWrapper<UserMeetTicket>()
                .eq(UserMeetTicket::getUserId, userId)
                .eq(UserMeetTicket::getStatus, UserMeetTicket.STATUS_AVAILABLE)
                .gt(UserMeetTicket::getExpireTime, LocalDateTime.now()));
        return list.stream().mapToInt(UserMeetTicket::getRemainAmount).sum();
    }

    private void minus(Long userId, Integer value) {
        if (value <= 0) {
            throw new IllegalArgumentException("数量错误");
        }
        LocalDateTime time = LocalDateTime.now();
        while (true) {
            UserMeetTicket meetTicket = baseMapper.selectUserAvailableTicket(userId, time);
            if (meetTicket == null) {
                throw new BusinessException("红豆不足");
            }
            Integer current = meetTicket.getRemainAmount();

            //当前的剩余值>=需要消耗的值
            if (baseMapper.optimismUpdateAmount(meetTicket.getId(), current - value, current) > 0) {
                return;
            } else {
                if (baseMapper.optimismUpdateAmount(meetTicket.getId(), 0, current) > 0) {
                    value = value - current;
                }
            }
        }
    }

    private void plus(Long userId, Integer value) {
        UserMeetTicket ticket = baseMapper.selectUserLastUsedTicket(userId, LocalDateTime.now());
        if (Objects.isNull(ticket)) {
            throw new BusinessException("Plus ticket is null");
        }
        boolean ret = baseMapper.optimismUpdateAmount(ticket.getId(), ticket.getRemainAmount() + value, ticket.getRemainAmount()) > 0;
        if (!ret) {
            throw new BusinessException("Ticket plus fail");
        }
    }

    private void saveRecord(Long userId, Integer amount, Long connectId, UserTicketRecordExt userTicketRecordExt, TicketRecordTypeEnum type) {
        ticketRecordService.save(new UserMeetTicketRecord()
                .setUserId(userId)
                .setAmount(amount)
                .setConnectId(connectId)
                .setType(type.getOrdinal())
                .setCategory(type.getCategory())
                .setExtJson(userTicketRecordExt));
    }

}
