package com.bxm.lovelink.common.dal.service.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.bxm.lovelink.common.autoconfigure.LoginApiOnlineProperties;
import com.bxm.lovelink.common.dal.service.IUserOnlineService;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author zhangdong
 * @date 2025/7/23
 */
@Service
@AllArgsConstructor
public class UserOnlineServiceImpl implements IUserOnlineService {
    private final Fetcher fetcher;
    private final Updater updater;
    private final JedisPool jedisPool;
    private final LoginApiOnlineProperties properties;


    @Override
    public Boolean getOnline(Long userId) {
        return compareOnline(getLastOnlineTime(userId));
    }

    @Override
    public void recordLastOnlineTime(Long userId) {
        updater.update(RedisKeys.userActiveTime(userId), LocalDateTimeUtil.format(LocalDateTime.now(), "yyyy-MM-dd HH:mm:ss"), properties.getKeepAliveTimeInSec());
    }

    @Override
    public Map<Long, Boolean> getBatchOnline(List<Long> userIds) {
        Map<Long, LocalDateTime> batchLastOnlineTime = getBatchLastOnlineTime(userIds);
        //调用compareOnline方法，并返回结果
        Map<Long, Boolean> result = new HashMap<>();
        batchLastOnlineTime.forEach((k, v) -> result.put(k, compareOnline(v)));
        return result;
    }

    @Override
    public LocalDateTime getLastOnlineTime(Long userId) {
        return parse(fetcher.fetch(RedisKeys.userActiveTime(userId), String.class));
    }

    @Override
    public Map<Long, LocalDateTime> getBatchLastOnlineTime(List<Long> userIds) {
        Map<Long, LocalDateTime> map = new HashMap<>();
        if (CollectionUtils.isEmpty(userIds)) {
            return map;
        }
        List<String> keys = Lists.newArrayList();
        userIds.forEach(x -> keys.add(RedisKeys.userActiveTime(x).generateKey()));
        List<String> values;
        try (Jedis jedis = jedisPool.getResource()) {
            values = jedis.mget(keys.toArray(new String[0]));
        }
        for (int i = 0; i < userIds.size(); i++) {
            map.put(userIds.get(i), parse(values.get(i)));
        }
        return map;
    }

    private LocalDateTime parse(String value) {
        if (StringUtils.isEmpty(value)) {
            return null;
        }
        return LocalDateTimeUtil.parse(value, "yyyy-MM-dd HH:mm:ss");
    }

    private Boolean compareOnline(LocalDateTime lastOnlineTime) {
        if (lastOnlineTime == null) {
            return false;
        }
        return LocalDateTimeUtil.between(lastOnlineTime, LocalDateTime.now(), ChronoUnit.SECONDS) <= properties.getOnlineTimeInSec();
    }
}
