package com.bxm.lovelink.common.dal.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.LoveLinkGlobalConfig;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.*;
import com.bxm.lovelink.common.dal.entity.dto.meetticket.TicketOperateDto;
import com.bxm.lovelink.common.dal.entity.dto.order.CreateOrderDto;
import com.bxm.lovelink.common.dal.entity.dto.order.PrepayDto;
import com.bxm.lovelink.common.dal.mapper.UserOrderMapper;
import com.bxm.lovelink.common.dal.service.*;
import com.bxm.lovelink.common.enums.TicketRecordTypeEnum;
import com.bxm.lovelink.common.event.promotion.UserActionEvent;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.lovelink.integration.apple.ApplePayIntegration;
import com.bxm.lovelink.integration.pay.PayIntegration;
import com.bxm.lovelink.integration.pay.facade.PayOrder;
import com.bxm.warcar.id.IdGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;

/**
 * <p>
 * 用户订单表 服务实现类
 * </p>
 *
 * @author tangxiao
 * @since 2025-05-15
 */
@Service
@Slf4j
public class UserOrderServiceImpl extends ServiceImpl<UserOrderMapper, UserOrder> implements IUserOrderService {

    private final IMeetTicketProductService meetTicketProductService;
    private final PayIntegration payIntegration;
    private final LoveLinkGlobalConfig loveLinkGlobalConfig;
    private final IUserBasicInfoService userBasicInfoService;
    private IUserMeetTicketService userMeetTicketService;
    private IdGenerator idGenerator;
    private final EventPark eventPark;
    private final IUserCertService userCertService;
    private final IActivityRegistrationService activityRegistrationService;

    public UserOrderServiceImpl(IMeetTicketProductService meetTicketProductService, PayIntegration payIntegration, LoveLinkGlobalConfig loveLinkGlobalConfig, IUserBasicInfoService userBasicInfoService, EventPark eventPark, IUserCertService userCertService, @Lazy IActivityRegistrationService activityRegistrationService) {
        this.meetTicketProductService = meetTicketProductService;
        this.payIntegration = payIntegration;
        this.loveLinkGlobalConfig = loveLinkGlobalConfig;
        this.userBasicInfoService = userBasicInfoService;
        this.eventPark = eventPark;
        this.userCertService = userCertService;
        this.activityRegistrationService = activityRegistrationService;
    }

    @Autowired
    @Qualifier("orderIdGenerator")
    public void setIdGenerator(IdGenerator idGenerator) {
        this.idGenerator = idGenerator;
    }

    // 解决循环依赖
    @Autowired
    public void setMeetTicketProductService(IUserMeetTicketService userMeetTicketService) {
        this.userMeetTicketService = userMeetTicketService;
    }

    @Override
    public UserOrder query(String orderNum) {
        return getOne(Wrappers.query(new UserOrder().setOrderNum(orderNum)));
    }

    @Override
    public String createOrder(CreateOrderDto dto, Long userId) {
        MeetTicketProduct product = meetTicketProductService.getById(dto.getMeetTicketProductId());
        if (product == null) {
            throw new IllegalArgumentException("商品不存在");
        }
        if (Objects.equals(product.getStatus(), Constants.STATUS_CLOSE)) {
            throw new IllegalArgumentException("商品已下架");
        }

        if (Objects.equals(product.getProductType(), Constants.MeetTicketProduct.CERT_PAY)) {
            if (StringUtils.isBlank(dto.getExtJson()) || !JSONObject.parseObject(dto.getExtJson()).containsKey("certType")) {
                throw new IllegalArgumentException("未找到certType");
            }
        }

        JSONObject ext = new JSONObject();
        if (Objects.equals(Constants.MeetTicketProduct.NORMAL, product.getProductType()) ||
                Objects.equals(Constants.MeetTicketProduct.VIP, product.getProductType())) {
            ext.put(Constants.UserOrder.ExtJsonKey.TICKET_VALID_DAYS, product.getTicketValidDays());
            if (product.getVipValidDays() != null) {
                LocalDateTime vipExpireTime = userBasicInfoService.computeVipExpireTime(userId, product.getVipValidDays());
                ext.put(Constants.UserOrder.ExtJsonKey.VIP_EXPIRE_TIME, vipExpireTime);
                ext.put(Constants.UserOrder.ExtJsonKey.VIP_VALID_DAYS, product.getVipValidDays());
            }
        }

        String orderNum = idGenerator.next();
        UserOrder order = new UserOrder()
                .setOrderNum(orderNum)
                .setUserId(userId)
                .setMeetTicketProductId(dto.getMeetTicketProductId())
                .setChannel(dto.getChannel())
                .setProductAmount(product.getAmount())
                .setProductType(product.getProductType())
                .setProductName(product.getName())
                .setProductDesc(product.getDescription())
                .setTotalPrice(product.getDiscountPrice())
                .setPaymentAmount(BigDecimal.ZERO)
                .setStatus(Constants.UserOrder.STATUS_WAIT_FOR_PAY)
                .setProductExtJson(ext.toJSONString())
                .setExtJson(dto.getExtJson());

        if (!save(order)) {
            throw new IllegalStateException("下单失败");
        }
        return orderNum;
    }

    @Override
    @Transactional(rollbackFor = Exception.class, timeout = 30)
    public Object prepay(PrepayDto prepayDto) {
        String orderNum = prepayDto.getOrderNum();
        UserOrder order = query(orderNum);
        if (Objects.isNull(order)) {
            throw new IllegalArgumentException("支付的订单不存在");
        }
        Integer status = order.getStatus();

        boolean isNotAvailablePrepayStatus = status > Constants.UserOrder.STATUS_PAYING;
        if (isNotAvailablePrepayStatus) {
            throw new IllegalStateException("订单状态不允许支付");
        }

        order.setPayType(prepayDto.getPayType());
        order.setStatus(Constants.UserOrder.STATUS_PAYING);

        if (!updateById(order)) {
            log.error("订单更新失败了 orderNum:{}", orderNum);
//            throw new IllegalStateException(orderNum + " 订单更新失败了");
        }
        if (prepayDto.isApplePay()) {
            return null;
        }

        PayOrder payOrder = new PayOrder()
                .setOrderNum(orderNum)
                .setAmount(order.getTotalPrice())
                .setPayType(prepayDto.getPayType())
                .setSubject(order.getProductName())
                .setBody(orderNum)
                .setClientType(prepayDto.getClientType())
                .setOpenId(prepayDto.getOpenId())
                .setClientIp(prepayDto.getClientIp())
                .setReboundUrl(prepayDto.getReboundUrl())
                .setNotifyUrl(loveLinkGlobalConfig.getPayNotifyUrl());
        return payIntegration.pay(payOrder);
    }

    @Override
    @Transactional(rollbackFor = Exception.class, timeout = 30)
    public void delivery(String orderNum, String paymentId, BigDecimal amount) {
        UserOrder order = query(orderNum);
        if (Objects.isNull(order)) {
            log.error("订单支付成功，交付阶段订单不存在：{}", orderNum);
            return;
        }

        UpdateWrapper<UserOrder> updateWrapper = new UpdateWrapper<>();
        updateWrapper
                .set(UserOrder.PAYMENT_ID, paymentId)
                .set(UserOrder.PAYMENT_AMOUNT, amount)
                .set(UserOrder.STATUS, Constants.UserOrder.STATUS_SUCCESS)
                .set(UserOrder.PAYMENT_TIME, LocalDateTime.now())
                .eq(UserOrder.ORDER_NUM, orderNum)
                .eq(UserOrder.STATUS, Constants.UserOrder.STATUS_PAYING);

        if (!update(updateWrapper)) {
            if (Constants.UserOrder.STATUS_PAYING != order.getStatus()) {
                log.warn("{} - 状态订单是 {}，不进行处理。", orderNum, order.getStatus());
                return;
            }
            throw new IllegalStateException(orderNum + " 订单更新失败了");
        }

        // 具体类型的后续处理
        Integer productType = order.getProductType();
        if (Constants.MeetTicketProduct.NORMAL == productType) {
            // 发放红豆
            this.saveUserMeetTicket(order);

            // 发送充值消费事件
            UserAction userAction = new UserAction();
            userAction.setUserId(order.getUserId());
            userAction.setType(UserAction.CONSUME);
            userAction.setAmount(amount);
            userAction.setOrderNum(orderNum);
            eventPark.post(new UserActionEvent(this, userAction));
        } else if (Constants.MeetTicketProduct.VIP == productType) {
            updateVipExpireTime(order);
        } else if (Constants.MeetTicketProduct.CERT_PAY == productType) {
            // 生成认证记录
            UserCert realNameCert = userCertService.getOneByUserIdAndType(order.getUserId(), UserCert.TYPE_REAL_NAME);
            JSONObject jsonObject = JSONObject.parseObject(order.getExtJson());
            String certType = jsonObject.getString("certType");
            UserCert userCert = new UserCert()
                    .setUserId(order.getUserId())
                    .setCertStatus(UserCert.STATUS_CHECKING)
                    .setCertType(Integer.parseInt(certType))
                    .setCertContent(realNameCert.getCertContent());
            userCertService.cert(userCert);
        } else if (Constants.MeetTicketProduct.ACTIVITY_PAY == productType) {
            ActivityRegistration activityRegistration = activityRegistrationService.getOneByOrderNo(order.getOrderNum());
            if (activityRegistration != null) {
                activityRegistration.setStatus(Constants.ActivityRegistrationStatus.SUCCESS);
                activityRegistrationService.updateById(activityRegistration);
            }

        }
    }

    @Override
    public void refund(Long id) {
        UserOrder order = getById(id);
        if (Objects.isNull(order)) {
            throw new BusinessException("订单不存在");
        }
        if (!order.isWxOrAliPay()) {
            throw new BusinessException("只允许支付宝或者微信退款");
        }
        if (!order.permitRefund()) {
            throw new BusinessException("当前支付状态下不允许退款");
        }
        boolean result = update(new LambdaUpdateWrapper<UserOrder>()
                .set(UserOrder::getStatus, Constants.UserOrder.STATUS_REFUNDING)
                .eq(UserOrder::getId, id).eq(UserOrder::getStatus, Constants.UserOrder.STATUS_SUCCESS));
        if (!result) {
            throw new BusinessException("订单修改中，请重试");
        }
        PayOrder payOrder = new PayOrder()
                .setOrderNum(order.getOrderNum())
                .setTradeNum(order.getPaymentId())
                .setRefundAmount(order.getPaymentAmount())
                .setRefundNotifyUrl(loveLinkGlobalConfig.getRefundNotifyUrl());
        payIntegration.refund(payOrder);
    }

    @Override
    public int rechargeOrderCount(Long userId) {
        QueryWrapper queryWrapper=new QueryWrapper();
        queryWrapper.eq(UserOrder.USER_ID,userId);
        queryWrapper.eq(UserOrder.STATUS,Constants.UserOrder.STATUS_SUCCESS);
        return count(queryWrapper);
    }

    private void saveUserMeetTicket(UserOrder order) {
        userMeetTicketService.rechargeTicket(new TicketOperateDto()
                        .setUserId(order.getUserId())
                        .setAmount(order.getProductAmount())
                        .setConnectId(order.getId())
                        .setType(order.isVipProduct() ? TicketRecordTypeEnum.RECHARGE_VIP : TicketRecordTypeEnum.RECHARGE_NORMAL));
    }

    @Override
    @Transactional(rollbackFor = Exception.class, timeout = 20)
    public void refundCallback(String orderNum, Boolean result) {
        UserOrder order = query(orderNum);
        if (Objects.isNull(order)) {
            log.error("退款回调订单不存在 orderNum:{}", orderNum);
            return;
        }
        int status = result ? Constants.UserOrder.STATUS_REFUND_SUCCESS : Constants.UserOrder.STATUS_REFUND_FAIL;
        boolean update = update(new LambdaUpdateWrapper<UserOrder>()
                .set(UserOrder::getStatus, status)
                .eq(UserOrder::getId, order.getId())
                .eq(UserOrder::getStatus, Constants.UserOrder.STATUS_REFUNDING));
        if (!update) {
            log.error("退款更新失败 order:{}", JSON.toJSONString(order));
        } else {
            // 发送退款事件
            UserAction userAction = new UserAction();
            userAction.setUserId(order.getUserId());
            userAction.setType(UserAction.REFUND);
            userAction.setAmount(order.getPaymentAmount());
            userAction.setOrderNum(order.getOrderNum());
            eventPark.post(new UserActionEvent(this, userAction));
        }
    }

    private void updateVipExpireTime(UserOrder order) {
        JSONObject extObject = JSONObject.parseObject(order.getProductExtJson());
        int validDays = extObject.getIntValue(Constants.UserOrder.ExtJsonKey.VIP_VALID_DAYS);

        userBasicInfoService.updateVipExpireTime(order.getUserId(), validDays);
    }

    @Override
    public boolean existsByTransactionId(String transactionId) {
        return count(new QueryWrapper<UserOrder>().eq(UserOrder.PAYMENT_ID, transactionId)) > 0;
    }

    @Override
    public UserOrder queryByTransactionId(String transactionId) {
        return getOne(new QueryWrapper<UserOrder>().eq(UserOrder.ORIGINAL_TRANSACTION_ID, transactionId));
    }

    @Override
    public UserOrder createIapOrder(Long userId, MeetTicketProduct product, ApplePayIntegration.VerifyReceiptResponse.InApp transaction) {
        JSONObject ext = new JSONObject();
        if (Objects.equals(Constants.MeetTicketProduct.NORMAL, product.getProductType()) ||
                Objects.equals(Constants.MeetTicketProduct.VIP, product.getProductType())) {
            ext.put(Constants.UserOrder.ExtJsonKey.TICKET_VALID_DAYS, product.getTicketValidDays());
            if (product.getVipValidDays() != null) {
                LocalDateTime vipExpireTime = userBasicInfoService.computeVipExpireTime(userId, product.getVipValidDays());
                ext.put(Constants.UserOrder.ExtJsonKey.VIP_EXPIRE_TIME, vipExpireTime);
                ext.put(Constants.UserOrder.ExtJsonKey.VIP_VALID_DAYS, product.getVipValidDays());
            }
        }

        String orderNum = idGenerator.next();
        UserOrder order = new UserOrder()
                .setOrderNum(orderNum)
                .setUserId(userId)
                .setMeetTicketProductId(product.getId())
                .setChannel("AppStore")
                .setProductAmount(product.getAmount())
                .setProductType(product.getProductType())
                .setProductName(product.getName())
                .setProductDesc(product.getDescription())
                .setTotalPrice(product.getDiscountPrice())
                .setPaymentAmount(BigDecimal.ZERO)
                .setStatus(Constants.UserOrder.STATUS_PAYING) // Set to paying initially
                .setPayType(Constants.UserOrder.PayType.APPLE_PAY)
                .setProductExtJson(ext.toJSONString())
                .setPaymentId(transaction.getTransactionId());
//                .setOriginalTransactionId(transaction.getOriginalTransactionId());

        if (!save(order)) {
            throw new IllegalStateException("Failed to create IAP order");
        }
        return order;
    }

    @Override
    public List<UserOrder> getPendingIapOrders() {
        return list(new QueryWrapper<UserOrder>()
                .eq(UserOrder.PAY_TYPE, Constants.UserOrder.PayType.APPLE_PAY)
                .in(UserOrder.STATUS, Constants.UserOrder.STATUS_WAIT_FOR_PAY, Constants.UserOrder.STATUS_PAYING));
    }
}
