package com.bxm.lovelink.common.dal.service.impl;

import cn.hutool.core.util.IdUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bxm.lovelink.common.cache.UserBasicInfoTableCache;
import com.bxm.lovelink.common.cache.UserLoveConditionTableCache;
import com.bxm.lovelink.common.cache.UserOtherTableCache;
import com.bxm.lovelink.common.cache.UserTableCache;
import com.bxm.lovelink.common.dal.entity.*;
import com.bxm.lovelink.common.dal.entity.dto.user.UserCompositeDto;
import com.bxm.lovelink.common.dal.entity.vo.user.UserBasicInfoVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserCompositeVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserLoveConditionVo;
import com.bxm.lovelink.common.dal.entity.vo.user.UserTagVo;
import com.bxm.lovelink.common.dal.mapper.UserMapper;
import com.bxm.lovelink.common.dal.mapping.UserInfoMapping;
import com.bxm.lovelink.common.dal.mapping.UserTagMapping;
import com.bxm.lovelink.common.dal.service.*;
import com.bxm.lovelink.common.event.promotion.UserActionEvent;
import com.bxm.lovelink.common.event.user.UserCancelEvent;
import com.bxm.lovelink.common.integration.uni.user.UniUserIntegration;
import com.bxm.lovelink.common.integration.uni.user.UniUserRegisterResponse;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import redis.clients.jedis.JedisPool;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 用户账号信息 服务实现类
 * </p>
 *
 * @author zhangdong
 * @since 2025-05-15
 */
@Service
@AllArgsConstructor
public class UserServiceImpl extends ServiceImpl<UserMapper, User> implements IUserService {

    private final IUserBasicInfoService userBasicInfoService;
    private final IUserOtherInfoService userOtherInfoService;
    private final IUserLoveConditionService userLoveConditionService;
    private final IUserCertService userCertService;
    private final IUserTagService userTagService;
    private final Fetcher fetcher;
    private final IRegionService regionService;
    private final RedissonClient redissonClient;
    private final IUserAuditService userAuditService;
    private final JedisPool jedisPool;
    private final IUserCmsService userCmsService;
    private final EventPark eventPark;
    private final UniUserIntegration uniUserIntegration;
    private final UserBasicInfoTableCache userBasicInfoCache;
    private final UserOtherTableCache userOtherCache;
    private final UserLoveConditionTableCache userLoveConditionCache;
    private final UserTableCache userTableCache;
    private final IUserOnlineService userOnlineService;

    @Override
    public UserComposite getCompositeById(Long id) {
        User user = this.getById(id);
        if (user == null) {
            return null;
        }
        return new UserComposite()
                .setUser(user)
                .setUserBasicInfo(userBasicInfoService.getOneByUserId(id))
                .setUserLoveCondition(userLoveConditionService.getOneByUserId(id))
                .setUserOtherInfo(userOtherInfoService.getOneByUserId(id))
                .setUserCms(userCmsService.getOneByUserId(id))
                .setUserCerts(userCertService.getListNewByUserId(id));
    }

    @Override
    public UserComposite getCompositeByUserNumber(String userNumber) {
        User user = this.getOne(new LambdaQueryWrapper<User>().eq(User::getUserNumber, userNumber));
        if (user == null) {
            return null;
        }
        return new UserComposite()
                .setUser(user)
                .setUserBasicInfo(userBasicInfoCache.getFromAuto(user.getId()))
                .setUserLoveCondition(userLoveConditionCache.getFromAuto(user.getId()))
                .setUserOtherInfo(userOtherCache.getFromAuto(user.getId()));
    }

    @Override
    public List<UserComposite> getCompositePartByUserIdsFromCache(List<Long> userIds) {
        Map<Long, User> userMap = userTableCache.getBatchFromAuto(userIds).stream()
                .collect(Collectors.toMap(User::getId, v -> v));
        Map<Long, UserBasicInfo> basicInfoMap = userBasicInfoCache.getBatchFromAuto(userIds).stream()
                .collect(Collectors.toMap(UserBasicInfo::getUserId, v -> v));
        Map<Long, UserLoveCondition> loveConditionMap = userLoveConditionCache.getBatchFromAuto(userIds).stream()
                .collect(Collectors.toMap(UserLoveCondition::getUserId, v -> v));
        Map<Long, UserOtherInfo> otherInfoMap = userOtherCache.getBatchFromAuto(userIds).stream()
                .collect(Collectors.toMap(UserOtherInfo::getUserId, v -> v));
        List<UserComposite> composites = new ArrayList<>();
        for (Long userId : userIds) {
            User user = userMap.get(userId);
            UserBasicInfo basicInfo = basicInfoMap.get(userId);
            UserLoveCondition loveCondition = loveConditionMap.get(userId);
            UserOtherInfo otherInfo = otherInfoMap.get(userId);
            if (user == null || basicInfo == null || loveCondition == null || otherInfo == null) {
                continue;
            }
            UserComposite composite = new UserComposite()
                    .setUser(user)
                    .setUserBasicInfo(basicInfo)
                    .setUserLoveCondition(loveCondition)
                    .setUserOtherInfo(otherInfo);
            composites.add(composite);
        }
        return composites;
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public Long createUser(User user) {
        String uniqueKey;
        if (StringUtils.isNotBlank(user.getUnionid())) {
            uniqueKey = user.getUnionid();
        } else if (StringUtils.isNotBlank(user.getAppleUserId())) {
            uniqueKey = user.getAppleUserId();
        } else {
            uniqueKey = user.getMobile();
        }
        RLock lock = redissonClient.getLock(RedisKeys.userCreateLock(uniqueKey).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            throw new IllegalArgumentException("注册中，请稍后再试");
        }
        try {
            if (StringUtils.isNotBlank(user.getUnionid())) {
                if (count(new LambdaQueryWrapper<User>().eq(User::getUnionid, user.getUnionid())) > 0) {
                    throw new IllegalArgumentException("该微信已经注册，请直接登录");
                }
            } else if (StringUtils.isNotBlank(user.getAppleUserId())) {
                if (count(new LambdaQueryWrapper<User>().eq(User::getAppleUserId, user.getAppleUserId())) > 0) {
                    throw new IllegalArgumentException("该苹果账号已经注册，请直接登录");
                }
            } else {
                if (count(new LambdaQueryWrapper<User>().eq(User::getMobile, user.getMobile())) > 0) {
                    throw new IllegalArgumentException("该手机号码已经注册，请直接登录");
                }
            }
            user.setUserNumber(IdUtil.fastSimpleUUID());
            save(user);
            userBasicInfoService.save(new UserBasicInfo().setUserId(user.getId()));
            userOtherInfoService.save(new UserOtherInfo().setUserId(user.getId()));
            userLoveConditionService.save(new UserLoveCondition().setUserId(user.getId()));
            userCmsService.save(new UserCms().setUserId(user.getId()));
            UniUserRegisterResponse register = uniUserIntegration.register(user.getId());
            user.setUniUserId(register.getUid());
            updateById(user);
            return user.getId();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void registerMobile(Long userId, String mobile) {
        RLock lock = redissonClient.getLock(RedisKeys.userCreateLock(mobile).generateKey());
        boolean b = lock.tryLock();
        if (!b) {
            throw new IllegalArgumentException("手机号码注册中，请稍后再试");
        }
        try {
            if (count(new LambdaQueryWrapper<User>().eq(User::getMobile, mobile)) > 0) {
                throw new IllegalArgumentException("手机号码已经注册，请换一个手机号");
            }
            updateById(new User().setMobile(mobile).setId(userId));
        } finally {
            lock.unlock();
        }

        // 发送激活推广事件
        eventPark.post(new UserActionEvent(this, new UserAction().setUserId(userId).setType(UserAction.ACTIVATION)));
    }

    @Override
    public UserCompositeVo getCompositeVoById(Long id) {
        UserComposite composite = getCompositeById(id);
        UserCompositeVo convert = convert(composite);
        UserAudit lastOneAudit = userAuditService.getLastOneAudit(new UserAudit()
                .setConnectId(composite.getUserBasicInfo().getId())
                .setUserId(id)
                .setAuditBusiness(UserAudit.AUDIT_BUSINESS_AVATAR));
        if (lastOneAudit != null) {
            convert.setAvatarAuditStatus(lastOneAudit.getAuditStatus());
        }
        return convert;
    }

    @Override
    public UserCompositeVo getCompositeVoByUserNumber(String userNumber) {
        UserComposite composite = getCompositeByUserNumber(userNumber);
        return convert(composite);
    }

    @Override
    public UserCompositeVo convert(UserComposite composite) {
        if (composite == null) {
            return null;
        }
        Long id = composite.getUser().getId();
        UserCompositeVo vo = UserInfoMapping.INSTANCE.doToVo(composite);
        UserBasicInfoVo basicInfo = vo.getUserBasicInfo();
        UserLoveConditionVo loveCondition = vo.getUserLoveCondition();
        //查询兴趣标签
        if (CollectionUtils.isNotEmpty(basicInfo.getInterestTag())) {
            basicInfo.setInterestTagVo(UserTagMapping.INSTANCE.doToVo(userTagService.listByIds(basicInfo.getInterestTag())));
        }
        //查询省市区信息
        ArrayList<Integer> cityCodes = Lists.newArrayList(basicInfo.getCurrentPlaceDistrict(), basicInfo.getNativePlaceDistrict(), basicInfo.getHouseCity(), basicInfo.getLastLoginDistrict());
        if (CollectionUtils.isNotEmpty(loveCondition.getNativePlaceRange())) {
            cityCodes.addAll(loveCondition.getNativePlaceRange());
        }
        Map<Integer, String> codeMap = regionService.queryByCodes(cityCodes);
        fillRegionName(vo, codeMap);
        //查询活跃状态
        vo.setActive(userOnlineService.getOnline(id));
        return vo;
    }

    @Override
    public List<UserCompositeVo> convert(List<UserComposite> composite) {
        List<UserCompositeVo> vos = UserInfoMapping.INSTANCE.doToVo(composite);
        if (CollectionUtils.isEmpty(composite)) {
            return vos;
        }
        List<Long> tagIds = Lists.newArrayList();
        List<Long> userIds = Lists.newArrayList();
        vos.forEach(c -> {
            userIds.add(c.getUser().getId());
            UserBasicInfoVo basicInfo = c.getUserBasicInfo();
            if (CollectionUtils.isNotEmpty(basicInfo.getInterestTag())) {
                tagIds.addAll(basicInfo.getInterestTag());
            }
        });
        //查询标签信息
        Map<Long, UserTagVo> tagMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(tagIds)) {
            tagMap = userTagService.fetchAll(UserTagMapping.INSTANCE::doToVo);
        }
        //查询省市区信息
        Map<Integer, String> codeMap = regionService.fetchAll(Region::getName);
        //查询活跃状态
        Map<Long, Boolean> onlineMap = userOnlineService.getBatchOnline(userIds);
        //组装数据
        Map<Long, UserTagVo> finalTagMap = tagMap;
        vos.forEach(c -> {
            UserBasicInfoVo basicInfo = c.getUserBasicInfo();
            fillRegionName(c, codeMap);
            if (CollectionUtils.isNotEmpty(basicInfo.getInterestTag())) {
                basicInfo.setInterestTagVo(basicInfo.getInterestTag().stream().map(finalTagMap::get).collect(Collectors.toList()));
            }
            c.setActive(onlineMap.get(basicInfo.getUserId()));
        });
        return vos;
    }

    private void fillRegionName(UserCompositeVo vo, Map<Integer, String> codeMap) {
        if (vo == null || codeMap == null || codeMap.size() == 0) {
            return;
        }
        UserBasicInfoVo basicInfo = vo.getUserBasicInfo();
        UserLoveConditionVo loveCondition = vo.getUserLoveCondition();
        basicInfo.setCurrentPlaceProvinceName(codeMap.get(basicInfo.getCurrentPlaceProvince()));
        basicInfo.setCurrentPlaceCityName(codeMap.get(basicInfo.getCurrentPlaceCity()));
        basicInfo.setCurrentPlaceDistrictName(codeMap.get(basicInfo.getCurrentPlaceDistrict()));
        basicInfo.setNativePlaceProvinceName(codeMap.get(basicInfo.getNativePlaceProvince()));
        basicInfo.setNativePlaceCityName(codeMap.get(basicInfo.getNativePlaceCity()));
        basicInfo.setNativePlaceDistrictName(codeMap.get(basicInfo.getNativePlaceDistrict()));
        basicInfo.setHouseCityName(codeMap.get(basicInfo.getHouseCity()));
        basicInfo.setHouseProvinceName(codeMap.get(basicInfo.getHouseProvince()));
        basicInfo.setLastLoginDistrictName(codeMap.get(basicInfo.getLastLoginDistrict()));
        if (CollectionUtils.isNotEmpty(loveCondition.getNativePlaceRange())) {
            loveCondition.setNativePlaceNameRange(loveCondition.getNativePlaceRange().stream().map(codeMap::get).collect(Collectors.toList()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateByCompositeDto(UserCompositeDto dto, UserComposite composite) {
        userBasicInfoService.updateByDto(dto.getUserBasicInfo(), composite);
        userLoveConditionService.updateByDto(dto.getUserLoveCondition(), composite);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void cancel(UserComposite userComposite, String reason) {
        Long userId = userComposite.getUser().getId();
        updateById(new User()
                .setId(userId)
                .setCancelReason(reason));
        removeById(userId);
        userBasicInfoService.removeById(userComposite.getUserBasicInfo().getId());
        userLoveConditionService.removeById(userComposite.getUserBasicInfo().getId());
        userCmsService.removeById(userComposite.getUserCms().getId());
        userOtherInfoService.removeById(userComposite.getUserOtherInfo().getId());
        eventPark.post(new UserCancelEvent(this, userId, userComposite));
    }


}
