package com.bxm.lovelink.common.dal.service.impl.huawei;

import com.bxm.lovelink.common.autoconfigure.HuaweiProperties;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.dto.meet.VirtualPhoneNumberBindResult;
import com.bxm.lovelink.common.dal.service.impl.AbstractVirtualPhoneNumberService;
import com.bxm.lovelink.common.dal.service.impl.huawei.impl.StringUtil;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.integration.pair.Value;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Random;

/**
 * @author Allen Hu
 * @date 2025/5/19
 */
@Slf4j
@Service
public class HuaweiVirtualPhoneNumberServiceImpl extends AbstractVirtualPhoneNumberService {

    private final HttpClient httpClient;
    private final String appKey;
    private final String appSecret;
    private final Pair pair;

    public HuaweiVirtualPhoneNumberServiceImpl(HttpClient httpClient, HuaweiProperties huaweiProperties, Pair pair, Fetcher fetcher, Updater updater) {
        super(fetcher, updater);
        this.httpClient = httpClient;
        this.appKey = huaweiProperties.getOmpAppKey();
        this.appSecret = huaweiProperties.getOmpAppSecret();
        this.pair = pair;
    }


    @Override
    public VirtualPhoneNumberBindResult bind(String a, String b) {
        HttpPost post = new HttpPost("https://rtcpns.cn-north-1.myhuaweicloud.com/rest/caas/relationnumber/partners/v1.0");
        try {
            AXBBindRequest request = new AXBBindRequest();
            request.setCallerNum(appendCountryCodeIfNeeded(a));
            request.setCalleeNum(appendCountryCodeIfNeeded(b));
            request.setRelationNum(appendCountryCodeIfNeeded(randomRelationNumber()));
            request.setDuration(getExpireTimeInSecond());

            post.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
            post.addHeader(HttpHeaders.AUTHORIZATION, "AKSK realm=\"SDP\",profile=\"UsernameToken\",type=\"Appkey\"");
            post.addHeader("X-AKSK", StringUtil.buildAKSKHeader(appKey, appSecret));

            post.setEntity(new StringEntity(JsonHelper.convert(request)));

            HttpResponse httpResponse = httpClient.execute(post);
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                log.error("隐私号码保护绑定失败！{}", statusCode);
                throw new BusinessException("隐私号码保护绑定失败！");
            }
            String body = EntityUtils.toString(httpResponse.getEntity());
            AXBBindResponse response = JsonHelper.convert(body, AXBBindResponse.class);
            if (!response.isSuccess()) {
                log.error("隐私号码保护绑定失败！{}", body);
                throw new BusinessException("隐私号码保护绑定失败！");
            }

            return new VirtualPhoneNumberBindResult()
                    .setVirtualNumber(response.getRelationNum())
                    .setSubscriptionId(response.getSubscriptionId());
        } catch (Exception e) {
            log.error("隐私号码保护绑定异常:", e);
            if (e instanceof BusinessException) {
                throw (BusinessException) e;
            }
            throw new RuntimeException(e);
        } finally {
            post.releaseConnection();
        }
    }

    @Override
    public void unbind(String subscriptionId) {
        String url = String.format("https://rtcpns.cn-north-1.myhuaweicloud.com/rest/caas/relationnumber/partners/v1.0?subscriptionId=%s", subscriptionId);
        HttpDelete delete = new HttpDelete(url);
        try {
            delete.addHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE);
            delete.addHeader(HttpHeaders.AUTHORIZATION, "AKSK realm=\"SDP\",profile=\"UsernameToken\",type=\"Appkey\"");
            delete.addHeader("X-AKSK", StringUtil.buildAKSKHeader(appKey, appSecret));

            HttpResponse httpResponse = httpClient.execute(delete);
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                log.error("隐私号码保护解绑失败！{}", statusCode);
                throw new BusinessException("隐私号码保护解绑失败！");
            }
            String body = EntityUtils.toString(httpResponse.getEntity());
            AXBUnbindResponse response = JsonHelper.convert(body, AXBUnbindResponse.class);
            if (!response.isSuccess()) {
                log.error("隐私号码保护解绑失败！{}", body);
                throw new BusinessException("隐私号码保护解绑失败！");
            }
        } catch (Exception e) {
            log.error("隐私号码保护解绑异常:", e);
            throw new RuntimeException(e);
        } finally {
            delete.releaseConnection();
        }
    }

    private String appendCountryCodeIfNeeded(String number) {
        return number.startsWith("+") || number.startsWith("0") ? number : "+86" + number;
    }

    private String randomRelationNumber() {
        Value value = pair.get(Constants.PairKey.HUAWEI_OMP_RELATION_NUMBERS);
        List<String> list = value.ofArrayList();
        if (list == null || list.isEmpty()) {
            log.warn("获取关系号码失败，请配置：{}", Constants.PairKey.HUAWEI_OMP_RELATION_NUMBERS);
            throw new BusinessException("获取关系号码失败");
        }
        return list.get(new Random().nextInt(list.size()));
    }
}
