package com.bxm.lovelink.common.dal.strategy;


import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.constant.RedisKeys;
import com.bxm.lovelink.common.dal.entity.InsertCard;
import com.bxm.lovelink.common.dal.entity.InsertCardContext;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.JsonHelper;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;
import java.util.Set;


/**
 * @author: weixing
 */
@Component
public class GuideCardInsertCardStrategy extends AbstractRewardInsertCardStrategy {
    private final Pair pair;
    private final Fetcher fetcher;
    private final Updater updater;

    public GuideCardInsertCardStrategy(Pair pair, Fetcher fetcher, Updater updater) {
        this.pair = pair;
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public Integer getCardType() {
        return Constants.CardType.GUIDE_CARD;
    }

    @Override
    public List<InsertCard> computeInsertCards(InsertCardContext context) {
        List<InsertCard> result = new ArrayList<>();
        List<Integer> pool = getAvailableCardPool(context);
        if (pool.isEmpty()) {
            return result;
        }
        Integer selectedCard = pool.get(new Random().nextInt(pool.size()));
        int positionRemainder = getPositionRemainder();
        int todaySwipeCount = context.getTodaySwipeCount() + ObjectUtils.defaultIfNull(context.getWaitSwipeCount(), 0);
        Integer requestSize = context.getRequestSize();
        int start = todaySwipeCount + 1;
        int end = todaySwipeCount + requestSize;
        for (int point = start; point <= end; point++) {
            if (point % 10 == positionRemainder) {
                int insertIndex = point - todaySwipeCount - 1;
                result.add(new InsertCard(insertIndex, getCardType(), selectedCard, Constants.SwipeEffect.ALLOW_SWIPE));
            }
        }
        return result;
    }

    private List<Integer> getAvailableCardPool(InsertCardContext context) {
        String date = LocalDate.now().format(DateTimeFormatter.BASIC_ISO_DATE);
        List<Integer> pool = new ArrayList<>();
        if (!context.getUserInfoComplete()) {
            pool.add(Constants.CardSubType.USER_INFO_COMPLETE);
        }
        Set<Integer> cards = fetcher.sfetchAll(RedisKeys.userTodayExposeGuideCard(context.getUserId(), date), Integer.class);
        pool.removeIf(cards::contains);
        return pool;

    }

    private int getPositionRemainder() {
        int positionRemainder = 8;
        String insertCardConfigStr = pair.get(Constants.PairKey.INSERT_CARD_CONFIG).of();
        if (StringUtils.isBlank(insertCardConfigStr)) {
            return positionRemainder;
        }
        InsertCardConfig insertCardConfig = JsonHelper.convert(insertCardConfigStr, InsertCardConfig.class);
        if (Objects.isNull(insertCardConfig) || MapUtils.isEmpty(insertCardConfig.getCardConfigMap())) {
            return positionRemainder;
        }
        Map<Integer, InsertCardConfig.CardConfig> cardConfigMap = insertCardConfig.getCardConfigMap();
        InsertCardConfig.CardConfig cardConfig = cardConfigMap.get(getCardType());
        if (Objects.nonNull(cardConfig) && Objects.nonNull(cardConfig.getPositionRemainder())) {
            return cardConfig.getPositionRemainder();
        }
        return positionRemainder;
    }

    @Override
    public Integer getSort() {
        return 3;
    }
}
