package com.bxm.lovelink.common.dal.strategy;


import com.bxm.lovelink.common.dal.entity.InsertCard;
import com.bxm.lovelink.common.dal.entity.InsertCardContext;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author: weixing
 */
@Component
public class InsertCardStrategyFactory implements ApplicationListener<ApplicationReadyEvent> {
    private final Map<Integer, InsertCardStrategy> strategyMap = new ConcurrentHashMap<>();

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        ConfigurableApplicationContext applicationContext = event.getApplicationContext();
        applicationContext.getBeansOfType(InsertCardStrategy.class).values().forEach(strategy -> {
            strategyMap.put(strategy.getCardType(), strategy);
        });
    }

    public InsertCardStrategy getStrategy(int type) {
        return strategyMap.get(type);
    }

    /**
     * 聚合所有策略结果，避免重复插入位置
     */
    public List<InsertCard> computeAllInsertCards(InsertCardContext context) {
        List<InsertCard> result = new ArrayList<>();
        Set<Integer> insertedIndexSet = new HashSet<>();

        List<InsertCardStrategy> strategies = new ArrayList<>(strategyMap.values());
        strategies.sort(Comparator.comparingInt(InsertCardStrategy::getSort));

        for (InsertCardStrategy strategy : strategies) {
            for (InsertCard card : strategy.computeInsertCards(context)) {
                if (insertedIndexSet.add(card.getIndex())) {
                    result.add(card);
                }
            }
        }

        // 倒序插入更安全（不影响下标）
        result.sort(Comparator.comparingInt(InsertCard::getIndex).reversed());
        return result;
    }
}
