package com.bxm.lovelink.common.dal.user.basicinfo;

import com.bxm.lovelink.common.dal.entity.UserAlbum;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.service.ISensitiveService;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.lovelink.common.integration.sensitive.SensitiveImageResult;
import com.bxm.lovelink.common.integration.sensitive.SensitiveImageType;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * 个人图片构建器
 *
 * @author zhangdong
 * @date 2025/5/17
 */
@Component
public class UserBasicInfoAlbumBuilder implements UserBasicInfoBuilder {

    private final ISensitiveService sensitiveService;

    private final ThreadPoolExecutor executor;

    public UserBasicInfoAlbumBuilder(ISensitiveService sensitiveService) {
        this.sensitiveService = sensitiveService;
        this.executor = new ThreadPoolExecutor(20, 200, 1L, TimeUnit.MINUTES,
                new LinkedBlockingQueue<>(), new NamedThreadFactory("user_album_audit"));
    }

    @Override
    public void buildAndCheck(UserBasicInfoBuilderContext context, UserBasicInfo userBasicInfo) {
        List<UserAlbum> newAlbum = userBasicInfo.getAlbum();
        if (CollectionUtils.isEmpty(newAlbum)) {
            return;
        }
        UserBasicInfo existBasicInfo = context.getExistBasicInfo();
        List<UserAlbum> existAlbum = existBasicInfo.getAlbum();
        if (existAlbum == null) {
            existAlbum = new ArrayList<>();
        }
        List<String> md5s = existAlbum.stream().map(UserAlbum::getMd5).collect(Collectors.toList());
        List<String> needCheck = new ArrayList<>();
        for (UserAlbum album : newAlbum) {
            if (!md5s.contains(album.getMd5())) {
                needCheck.add(album.getUrl());
            }
        }
        if (CollectionUtils.isEmpty(needCheck)) {
            return;
        }
        CountDownLatch countDownLatch = new CountDownLatch(needCheck.size());
        List<Future<SensitiveImageResult>> futures = new ArrayList<>();
        for (String url : needCheck) {
            futures.add(executor.submit(() -> {
                try {
                    return sensitiveService.checkImage(SensitiveImageType.LIFE, url);
                } finally {
                    countDownLatch.countDown();
                }
            }));
        }
        try {
            countDownLatch.await();
            for (Future<SensitiveImageResult> future : futures) {
                SensitiveImageResult result = future.get();
                if (result.getHit()) {
                    throw new IllegalArgumentException("照片内容不合规，请重新上传");
                }
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new IllegalArgumentException("照片检测异常，请重试");
        }
    }

    @Override
    public int getOrder() {
        return BuilderSortEnum.ALBUM_AUDIT.getCode();
    }
}
