package com.bxm.lovelink.common.enums;

import com.bxm.lovelink.common.contant.UserConstants;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.dal.entity.UserComposite;
import com.bxm.lovelink.common.dal.entity.UserRuleTag;
import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

/**
 * @author zhangdong
 * @date 2025/5/20
 */
@AllArgsConstructor
@Getter
public enum UserRuleTagEnum {

    EDU(1, "学历标签，本科及以上有", new Function<UserComposite, String>() {

        @Override
        public String apply(UserComposite userComposite) {
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            Integer edu = basicInfo.getEduHighest();
            return getRuleTagName(UserConstants.EduHighestEnum.BACHELOR, edu);
        }
    }),

    YEAR_REVENUE(2, "年收入标签，大于25万有年收入的标签，展示具体年收入区间", new Function<UserComposite, String>() {
        @Override
        public String apply(UserComposite userComposite) {
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            Integer yearRevenue = basicInfo.getYearRevenue();
            return getRuleTagName(UserConstants.YearRevenueEnum.FROM_20K_TO_30K, yearRevenue);
        }
    }),

    CAR(3, "需车认证通过", new Function<UserComposite, String>() {
        @Override
        public String apply(UserComposite userComposite) {
//            List<UserCert> userCerts = userComposite.getUserCerts();
//            if (CollectionUtils.isEmpty(userCerts)) {
//                return null;
//            }
//            userCerts.sort((o1, o2) -> o2.getId().compareTo(o1.getId()));
//            UserCert userCert = userCerts.stream().filter(x -> Objects.equals(x.getCertType(), UserCert.TYPE_CAR)).findFirst().orElse(null);
//            if (userCert == null || !userCert.isSuccess()) {
//                return null;
//            }
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            Integer car = basicInfo.getCar();
            if (UserConstants.CarEnum.HAS_CAR.getCode().equals(car)) {
                return "有车";
            }
            return null;
        }
    }),

    HOUSE(4, "需房产认证通过", new Function<UserComposite, String>() {
        @Override
        public String apply(UserComposite userComposite) {
//            List<UserCert> userCerts = userComposite.getUserCerts();
//            if (CollectionUtils.isEmpty(userCerts)) {
//                return null;
//            }
//            userCerts.sort((o1, o2) -> o2.getId().compareTo(o1.getId()));
//            UserCert userCert = userCerts.stream().filter(x -> Objects.equals(x.getCertType(), UserCert.TYPE_HOUSE)).findFirst().orElse(null);
//            if (userCert == null || !userCert.isSuccess()) {
//                return null;
//            }
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            Integer house = basicInfo.getHouse();
            if (UserConstants.HouseEnum.HAS_HOUSE.getCode().equals(house)) {
                return "有房";
            }
            return null;
        }
    }),

    SCHOOL(5, "学校", new Function<UserComposite, String>() {
        @Override
        public String apply(UserComposite userComposite) {
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            String schoolName = basicInfo.getSchoolName();
            if (StringUtils.isBlank(schoolName) || !basicInfo.isSchoolDisplay()) {
                return null;
            }
            return schoolName;
        }
    }),


    HEIGHT(6, "身高", new Function<UserComposite, String>() {
        @Override
        public String apply(UserComposite userComposite) {
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            Integer height = basicInfo.getHeight();
            if (height == null || basicInfo.getGender() == null) {
                return null;
            }
            if (basicInfo.isMan() && height >= 175) {
                return height + "cm";
            }
            if (!basicInfo.isMan() && height >= 160) {
                return height + "cm";
            }
            return null;
        }
    }),

    CONSTELLATION(7, "星座", new Function<UserComposite, String>() {
        @Override
        public String apply(UserComposite userComposite) {
            UserBasicInfo basicInfo = userComposite.getUserBasicInfo();
            Integer constellation = basicInfo.getConstellation();
            if (constellation == null) {
                return null;
            }
            UserConstants.UserConstellationEnum byCode = UserConstants.UserConstellationEnum.getByCode(constellation);
            if (byCode == null) {
                return null;
            }
            return byCode.getName();
        }
    }),


    HIGH_JOB(8, "高级职业", null),


    ;

    private final Integer type;

    private final String desc;
    private final Function<UserComposite, String> calculate;

    public static List<UserRuleTag> getUserRuleTags(UserComposite userComposite) {
        List<UserRuleTag> userRuleTags = new ArrayList<>();
        for (UserRuleTagEnum x : UserRuleTagEnum.values()) {
            if (x.calculate == null) {
                continue;
            }
            String apply = x.calculate.apply(userComposite);
            if (apply != null) {
                userRuleTags.add(new UserRuleTag().setType(x.getType()).setName(apply));
            }
        }
        return userRuleTags;
    }


    private static String getRuleTagName(UserConstants.BaseEnum ruleLine, Integer code) {
        if (code == null || ruleLine == null) {
            return null;
        }
        Integer ruleLineCode = ruleLine.getCode();
        if (code < ruleLineCode) {
            return null;
        }
        for (UserConstants.BaseEnum baseEnum : ruleLine.getClass().getEnumConstants()) {
            if (Objects.equals(baseEnum.getCode(), code)) {
                return baseEnum.getName();
            }
        }
        return null;
    }
}
