package com.bxm.lovelink.common.integration.sensitive.wangyi;

import com.alibaba.fastjson.JSON;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.lovelink.common.integration.sensitive.*;
import com.google.common.collect.Lists;
import com.netease.yidun.sdk.antispam.AntispamRequester;
import com.netease.yidun.sdk.antispam.image.v5.ImageClient;
import com.netease.yidun.sdk.antispam.image.v5.check.ImageV5CheckRequest;
import com.netease.yidun.sdk.antispam.image.v5.check.sync.request.ImageV5SyncCheckRequest;
import com.netease.yidun.sdk.antispam.image.v5.check.sync.response.ImageV5CheckResponse;
import com.netease.yidun.sdk.antispam.image.v5.check.sync.response.ImageV5Result;
import com.netease.yidun.sdk.antispam.text.TextClient;
import com.netease.yidun.sdk.antispam.text.v5.check.sync.single.TextCheckRequest;
import com.netease.yidun.sdk.antispam.text.v5.check.sync.single.TextCheckResponse;
import com.netease.yidun.sdk.antispam.text.v5.check.sync.single.TextCheckResult;
import com.netease.yidun.sdk.core.auth.Credentials;
import com.netease.yidun.sdk.core.client.ClientProfile;
import com.netease.yidun.sdk.core.http.HttpClientConfig;
import com.netease.yidun.sdk.core.response.CommonResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * 图片检测
 *
 * @author zhangdong
 * @date 2025/5/16
 */
@Slf4j
@Component
@ConditionalOnProperty(prefix = "sensitive.wang-yi", name = "enable", havingValue = "true")
public class WangYiSensitiveIntegration implements SensitiveImageIntegration, SensitiveTextIntegration {
    private final WangYiSensitiveProperties properties;
    private final ImageClient imageClient;
    private final TextClient textClient;

    public WangYiSensitiveIntegration(WangYiSensitiveProperties properties) {
        this.properties = properties;
        imageClient = ImageClient.getInstance(AntispamRequester.getInstance(createProfile()));
        textClient = TextClient.getInstance(AntispamRequester.getInstance(createProfile()));
    }

    @Override
    public SensitiveImageResult checkImage(Integer type, String imgUrl) {
        String businessId = properties.getAvatarBusinessId();
        if (Objects.equals(SensitiveImageType.LIFE, type)) {
            businessId = properties.getLifeBusinessId();
        }
        SensitiveImageResult result = new SensitiveImageResult()
                .setImgUrl(imgUrl);
        String errorMessage = "wangYi check image error";
        if (StringUtils.isEmpty(imgUrl)) {
            return result;
        }
        // 实例化请求对象
        ImageV5SyncCheckRequest checkRequest = new ImageV5SyncCheckRequest();
        // 设置易盾内容安全分配的businessId
        checkRequest.setBusinessId(businessId);
        // 实例化要检测的图片对象
        ImageV5CheckRequest.ImageBeanRequest image = new ImageV5CheckRequest.ImageBeanRequest();
        image.setData(imgUrl);
        image.setName("1");
        // 设置图片数据的类型，1：图片URL，2:图片BASE64
        image.setType(1);
        // 设置需要检测的图片列表，最大32张或者10M
        checkRequest.setImages(Lists.newArrayList(image));

        // 请求对象中的其他参数如果有需要，请参考官方接口文档中字段说明，按需添加
        ImageV5CheckResponse checkResponse;
        try {
            // 发起同步检测的请求
            checkResponse = imageClient.syncCheckImage(checkRequest);
        } catch (Exception e) {
            log.error(errorMessage + ",url:{}", imgUrl, e);
            throw new BusinessException(errorMessage);
        }
        if (fail(checkResponse)) {
            log.error(errorMessage + ",url:{},data:{}", imgUrl, JSON.toJSONString(checkResponse));
            throw new BusinessException(errorMessage);
        }
        ImageV5Result imageV5Result = checkResponse.getResult().get(0);
        result.setHit(!Objects.equals(imageV5Result.getAntispam().getSuggestion(), 0));
        return result;
    }

    @Override
    public SensitiveTextResult checkText(String text) {
        SensitiveTextResult textResult = new SensitiveTextResult().setText(text);
        if (StringUtils.isEmpty(text)) {
            return textResult;
        }
        // 实例化请求对象
        TextCheckRequest checkRequest = new TextCheckRequest();
        // 设置易盾内容安全分配的businessId
        checkRequest.setBusinessId(properties.getTextBusinessId());
        // 设置检测内容
        checkRequest.setDataId("1");
        checkRequest.setContent(text);
        // 请求对象中的其他参数如果有需要，请参考官方接口文档中字段说明，按需添加
        TextCheckResponse checkResponse;
        String errorMessage = "wangYi text check error";
        try {
            // 发起同步检测的请求
            checkResponse = textClient.syncCheckText(checkRequest);
        } catch (Exception e) {
            log.error(errorMessage + ",text:{}", text, e);
            throw new BusinessException(errorMessage);
        }
        if (fail(checkResponse)) {
            log.error(errorMessage + ",text:{},data:{}", text, JSON.toJSONString(checkResponse));
            throw new BusinessException(errorMessage);
        }
        // 获取文本的检测结果，具体返回字段的说明，请参考官方接口文档中字段说明
        TextCheckResult.Antispam antispam = checkResponse.getResult().getAntispam();
        if (!Objects.equals(antispam.getSuggestion(), 0)) {
            textResult.setHit(true);
            textResult.setText(antispam.getFilteredContent());
        }
        return textResult;
    }

    /**
     * 创建自定义请求配置
     */
    private ClientProfile createProfile() {
        ClientProfile clientProfile = ClientProfile.defaultProfile(new Credentials(properties.getSecretId(), properties.getSecretKey()));
        // 设置http请求的相关配置
        HttpClientConfig httpClientConfig = new HttpClientConfig();
        httpClientConfig.setMaxConnectionCountPerRoute(20);
        httpClientConfig.setMaxConnectionCount(20);
        httpClientConfig.setConnectionRequestTimeoutMillis(1000);
        httpClientConfig.setConnectionTimeoutMillis(2000);
        httpClientConfig.socketTimeoutMillis(3000);
        clientProfile.setHttpClientConfig(httpClientConfig);
        return clientProfile;
    }

    private boolean fail(CommonResponse response) {
        return Objects.isNull(response) || !Objects.equals(response.getCode(), 200);
    }


}
