package com.bxm.lovelink.common.integration.sms.submail;

import com.alibaba.fastjson.JSON;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.warcar.utils.HttpClientHelper;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import java.nio.charset.StandardCharsets;

/**
 * @author zhangdong
 * @date 2025/5/16
 */
@Slf4j
@Configuration
@EnableConfigurationProperties(SubMailSmsProperties.class)
public class SubMailSmsIntegration {
    private final HttpClient httpClient;
    private final SubMailSmsProperties properties;
    private static final String SMS_URL = "https://api-v4.mysubmail.com/sms/send.json";
    private static final String SMS_TEMPLATE_URL = "https://api-v4.mysubmail.com/sms/xsend.json";


    public SubMailSmsIntegration(SubMailSmsProperties properties) {
        this.properties = properties;
        this.httpClient = HttpClientHelper.createHttpClient(50, 50, 5000, 5000, 5000);
    }

    public void send(String to, String content) {
        SubMailSmsRequest message = new SubMailSmsRequest()
                .setAppid(properties.getAppid())
                .setSignature(properties.getSignature())
                .setTo(to)
                .setContent(content);
        doSend(message, SMS_URL);
    }

    public void sendByTemplate(String to, String templateId, String vars) {
        SubMailSmsRequest message = new SubMailSmsRequest()
                .setAppid(properties.getAppid())
                .setSignature(properties.getSignature())
                .setTo(to)
                .setProject(templateId)
                .setVars(vars);
        doSend(message, SMS_TEMPLATE_URL);
    }

    public void doSend(SubMailSmsRequest message, String url) {
        String reqInfo = JSON.toJSONString(message);
        HttpPost post = new HttpPost(url);
        post.setEntity(new StringEntity(JSON.toJSONString(message), StandardCharsets.UTF_8));
        post.addHeader("content-type", "application/json;charset=utf-8");
        String errorMessage = "message send error";
        try {
            HttpResponse response = httpClient.execute(post);
            int statusCode = response.getStatusLine().getStatusCode();
            String body = EntityUtils.toString(response.getEntity());
            if (statusCode != HttpStatus.SC_OK || StringUtils.isBlank(body)) {
                log.error(errorMessage + ",statusCode: {},message: {}, response: {}", statusCode, reqInfo, body);
                throw new BusinessException(errorMessage);
            }
            log.info("{} - {}", message.getTo(), body);
            SubMailSmsResponse rs = JsonHelper.convert(body, SubMailSmsResponse.class);
            if (!"success".equals(rs.getStatus())) {
                log.error(errorMessage + ",message:{}, response: {}", reqInfo, body);
                throw new BusinessException(errorMessage);
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            log.error(errorMessage + ",message:{}", reqInfo, e);
            throw new BusinessException(errorMessage);
        } finally {
            post.releaseConnection();
        }
    }
}
