package com.bxm.lovelink.common.integration.uni.user;

import com.alibaba.fastjson.JSON;
import com.bxm.lovelink.common.dal.entity.UserBasicInfo;
import com.bxm.lovelink.common.exception.BusinessException;
import com.bxm.lovelink.common.integration.uni.UniProperties;
import com.bxm.warcar.utils.HttpClientHelper;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;

import java.nio.charset.StandardCharsets;
import java.util.Objects;

/**
 * @author zhangdong
 * @date 2025/6/26
 */
@Configuration
@Slf4j
@EnableConfigurationProperties(UniProperties.class)
public class UniUserIntegration {

    private final HttpClient httpClient;
    private final UniProperties properties;

    public UniUserIntegration(UniProperties properties) {
        this.properties = properties;
        httpClient = HttpClientHelper.createHttpClient(50, 50, 5000, 5000, 5000);
    }


    public UniUserRegisterResponse register(Long userId) {
        UniUserRegisterRequest request = new UniUserRegisterRequest();
        request.setExternalUid(String.valueOf(userId));
        return doPost(request, new UniUserClientInfo(), UniUserRegisterResponse.class, "user/externalRegister");
    }

    public UniUserLoginResponse login(Long userId, Object clientInfo) {
        UniUserLoginRequest request = new UniUserLoginRequest();
        request.setExternalUid(String.valueOf(userId));
        return doPost(request, clientInfo, UniUserLoginResponse.class, "user/externalLogin");
    }

    public UniUserUpdateResponse update(UserBasicInfo userBasicInfo) {
        UniUserUpdateRequest request = new UniUserUpdateRequest();
        request.setExternalUid(String.valueOf(userBasicInfo.getUserId()));
        request.setNickname(userBasicInfo.getNickname());
        request.setAvatar(userBasicInfo.getAvatar());
        if (Objects.nonNull(userBasicInfo.getGender())) {
            request.setGender(userBasicInfo.getGender() == 1 ? 1 : 2);
        }
        return doPost(request, new UniUserClientInfo(), UniUserUpdateResponse.class, "user/updateUserInfoByExternal");
    }

    public UniUserUpdateResponse cancel(Long userId) {
        UniUserUpdateRequest request = new UniUserUpdateRequest();
        request.setExternalUid(String.valueOf(userId));
        request.setStatus(4);
        return doPost(request, new UniUserClientInfo(), UniUserUpdateResponse.class, "user/updateUserInfoByExternal");
    }


    public <T extends UniUserCommonResponse> T doPost(Object obj, Object clientInfo, Class<T> clazz, String uri) {
        UniUserCommonRequest request = new UniUserCommonRequest();
        if (clientInfo == null) {
            clientInfo = new UniUserClientInfo();
        }
        request.setParams(obj);
        request.setClientInfo(clientInfo);
        String reqInfo = JSON.toJSONString(request);
        HttpPost post = new HttpPost(properties.getDomain() + uri);
        post.setEntity(new StringEntity(reqInfo, StandardCharsets.UTF_8));
        post.addHeader("content-type", "application/json;charset=utf-8");
        post.addHeader("Cache-Control", "no-cache");
        post.addHeader(properties.getAuthHeaderName(), properties.getAuthHeaderValue());
        String errorMessage = "uni user error";
        try {
            HttpResponse response = httpClient.execute(post);
            int statusCode = response.getStatusLine().getStatusCode();
            String body = EntityUtils.toString(response.getEntity());
            if (statusCode != HttpStatus.SC_OK || StringUtils.isBlank(body)) {
                log.error(errorMessage + ",statusCode: {},message: {}, response: {}", statusCode, reqInfo, body);
                throw new BusinessException(errorMessage);
            }
            T rs = JsonHelper.convert(body, clazz);
            if (!rs.isSuccess()) {
                log.error(errorMessage + ",message:{}, response: {}", reqInfo, body);
                throw new BusinessException(errorMessage);
            }
            return rs;
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            log.error(errorMessage + ",message:{}", reqInfo, e);
            throw new BusinessException(errorMessage);
        } finally {
            post.releaseConnection();
        }
    }
}
