package com.bxm.lovelink.common.push;

import com.alibaba.fastjson.JSON;
import com.bxm.lovelink.common.dal.entity.PushTask;
import com.bxm.lovelink.common.dal.entity.User;
import com.bxm.lovelink.common.dal.service.IUserService;
import com.bxm.lovelink.common.integration.uni.push.UniappPushMsgRequest;
import com.bxm.lovelink.common.push.event.push.PushEvent;
import com.bxm.lovelink.common.push.factory.*;
import com.bxm.lovelink.common.push.render.PushRender;
import com.bxm.lovelink.common.push.render.PushRenderFactory;
import com.bxm.warcar.integration.eventbus.EventPark;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * 推送总控制器
 *
 * @author Allen Hu
 * @date 2025/6/17
 */
@Slf4j
@Component
@AllArgsConstructor
public class Pusher {

    private final PushSceneServiceFactory pushSceneServiceFactory;
    private final PushRenderFactory pushRenderFactory;
    private final EventPark eventPark;
    private final IUserService userService;

    @Async
    public void doTimerPush(PushScene pushScene) {
        TimerPushSceneService pushSceneService = pushSceneServiceFactory.getTimerPushSceneService(pushScene);
        if (pushSceneService == null) {
            log.warn("Unsupported Push Scene: {}", pushScene);
            return;
        }
        List<Push> targetResultList = pushSceneService.create();
        if(CollectionUtils.isEmpty(targetResultList)){
            log.warn("targetResultList is not config: {}", pushScene);
            return;
        }
        for (Push targetResult : targetResultList) {
            eventPark.post(new PushEvent(this, targetResult));
        }
        pushSceneService.onAfterPostEvent(targetResultList);
    }

    @Async
    public void doImmediatePush(PushScene pushScene, PushTarget pushTarget) {
        ImmediatePushSceneService pushSceneService = pushSceneServiceFactory.getImmediatePushSceneService(pushScene);
        if (pushSceneService == null) {
            log.warn("Unsupported Push Scene: {}", pushScene);
            return;
        }
        List<Push> targetResultList = pushSceneService.create(pushTarget);
        if(CollectionUtils.isEmpty(targetResultList)){
            log.warn("targetResultList is not config: {}", pushScene);
            return;
        }
        for (Push targetResult : targetResultList) {
            eventPark.post(new PushEvent(this, targetResult));
        }
    }

    public UniappPushMsgRequest buildUniappPushMsgRequest(Push push) {
        PushTask pushTask = push.getPushTask();
        PushTarget pushTarget = push.getPushTarget();
        String taskScene = pushTask.getTaskScene();
        PushScene pushScene = PushScene.valueOf(taskScene);
        Long userId = pushTarget.getUserId();
        User user = userService.getById(userId);
        if(Objects.isNull(user)){
            log.info("用户ID是空 {}", JSON.toJSONString(push));
            return null;
        }
        String pushClientid = user.getPushClientid();
        if (StringUtils.isBlank(pushClientid)) {
            log.info("没有 pushClinentid 的用户不推送消息,用户ID是 {}", userId);
            return null;
        }

        PushRender pushRender = Optional.ofNullable(pushRenderFactory.get(pushScene)).orElse(PushRender.SIMPLE);
        String message = pushRender.renderPushMessage(pushTask, pushTarget);
        String payload = pushRender.renderPushPayload(pushTask, pushTarget);
        Map<String, Object> payloadObject = Optional.ofNullable(pushTarget.getPayload())
                .orElseGet(() -> Collections.singletonMap("path", payload));
        Map<String, Object> extend = pushTarget.getExtend();
        return new UniappPushMsgRequest()
                .setPush_clientid(pushClientid)
                .setContent(message)
                .setTitle(pushTask.getPushTitle())
                .setPayload(payloadObject)
                .setForce_notification(Objects.isNull(extend.get("force_notification")) || (Boolean) extend.get("force_notification"));
    }
}
