package com.bxm.lovelink.common.push.factory;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bxm.lovelink.common.dal.entity.PushTask;
import com.bxm.lovelink.common.dal.service.IPushTaskService;
import com.bxm.lovelink.common.push.ExceedMaximumCountException;
import com.bxm.lovelink.common.push.ExceedMaximumLifeTimeException;
import com.bxm.lovelink.common.push.TaskRedisKeys;
import com.bxm.lovelink.common.push.render.PushRender;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * @author Allen Hu
 * @date 2025/6/17
 */
@Slf4j
public abstract class AbstractTimerPushSceneService implements TimerPushSceneService, PushRender {

    private final IPushTaskService pushTaskService;
    private final Fetcher fetcher;
    private final JedisPool jedisPool;

    protected AbstractTimerPushSceneService(IPushTaskService pushTaskService, Fetcher fetcher) {
        this.pushTaskService = pushTaskService;
        this.fetcher = fetcher;
        this.jedisPool = (JedisPool) fetcher.getClientOriginal();
    }

    /**
     * 根据推送任务查找符合条件的推送目标
     * @param pushTask 推送任务
     * @return 符合条件的推送目标
     */
    protected abstract List<PushTarget> findPushTaskTargets(PushTask pushTask);

    /**
     * 检查推送任务目标是否已终止
     *
     * @param pushTarget 推送任务目标
     * @param now 当前时间
     * @param taskEndOfLifeInDays 终止时间n填
     * @return 是否已终止
     */
    private boolean isTerminated(PushTarget pushTarget, LocalDateTime now, long taskEndOfLifeInDays) {
        LocalDateTime startTime = pushTarget.getStartTime();
        if (null == startTime) {
            return false;
        }
        return startTime.plusDays(taskEndOfLifeInDays).isBefore(now);
    }

    @Override
    public List<Push> create() {
        // 首先查询这个场景的任务列表
        QueryWrapper<PushTask> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(PushTask.TASK_SCENE, this.of().name());
        queryWrapper.eq(PushTask.STATUS, 1);
        List<PushTask> pushTasks = pushTaskService.list(queryWrapper);
        if (CollectionUtils.isEmpty(pushTasks)) {
            log.info("场景[{}]没有需要处理的任务", this.of().name());
            return Collections.emptyList();
        }
        List<Push> results = Lists.newArrayList();
        // 查询符合条件的目标集合
        for (PushTask pushTask : pushTasks) {
            List<PushTarget> pushTargets = findPushTaskTargets(pushTask);
            if (CollectionUtils.isEmpty(pushTargets)) {
                log.debug("PushTask:{} has no target", pushTask);
                continue;
            }
            List<Push> targetResultList = this.filter(pushTask, pushTargets);
            results.addAll(targetResultList);
        }
        return results;
    }

    private List<Push> filter(PushTask pushTask, List<PushTarget> pushTargets) {
        List<Push> results = Lists.newArrayListWithCapacity(pushTargets.size());
        LocalDateTime now = LocalDateTime.now();

        for (PushTarget pushTarget : pushTargets) {
            Long userId = pushTarget.getUserId();
            try {
                if (isCheckIfPushOnConditional()) {
                    // 检查目标是否允许推送
                    this.checkIfPushOnConditional(pushTask, pushTarget, now);
                }
                // 添加到结果集
                results.add(new Push(pushTask, pushTarget));
            } catch (ExceedMaximumLifeTimeException e) {
                log.debug("目标：{}，已经超过了 {} 最大推送时间，不再推送", userId, pushTask.getTaskScene());
            } catch (ExceedMaximumCountException e) {
                log.debug("目标：{}，已经超过了 {} 最大推送次数，不再推送", userId, pushTask.getTaskScene());
            }
        }
        return results;
    }

    /**
     * 检查是否满足推送条件
     * 
     * @param pushTask 推送认为 
     * @param pushTarget 推送目标
     * @throws ExceedMaximumLifeTimeException 超过最大生命周期
     * @throws ExceedMaximumCountException 超过了最大推送次数
     */
    private void checkIfPushOnConditional(PushTask pushTask, PushTarget pushTarget, LocalDateTime now) throws ExceedMaximumLifeTimeException, ExceedMaximumCountException {
        // 检查目标是否已经超过了任务的终止期
        Long days = pushTask.getTaskEndOfLife();
        if (isTerminated(pushTarget, now, days)) {
            throw new ExceedMaximumLifeTimeException();
        }
        
        // 检查目标在当前周期里是否已经达到最大次数
        Integer taskCountRange = pushTask.getTaskCountRange();
        // 默认是ID范围
        boolean sceneRange = taskCountRange == 2;

        Integer pushMaximumCount = pushTask.getPushMaximumCount();

        // 检查目标当前是否处于沉默期
        Integer pushCycleHours = pushTask.getPushCycleHours();

        Long userId = pushTarget.getUserId();
        KeyGenerator counterKey = TaskRedisKeys.stringPushCount(pushTask.getId(), userId);
        if (sceneRange) {
            counterKey = TaskRedisKeys.stringPushCount(pushTask.getTaskScene(), userId);
        }

        // 判断已经推送的次数是否已经达到推送限制
        Long currentCount = Optional.ofNullable(fetcher.fetch(counterKey, Long.class)).orElse(0L);
        if (currentCount >= pushMaximumCount) {
            // 已经超过了最大推送次数
            throw new ExceedMaximumCountException();
        }

        // 设置计数和沉默期
        // TODO 后续流程失败，不会回滚计数，所以导致丢失推送。
        this.increment(counterKey, pushCycleHours);
    }

    protected void increment(KeyGenerator counterKey, Integer pushCycleHours) {
        try (Jedis jedis = jedisPool.getResource()) {
            String key = counterKey.generateKey();
            Boolean exists = jedis.exists(key);
            jedis.incr(key);
            if (!exists) {
                long seconds = Duration.ofHours(pushCycleHours).getSeconds();
                jedis.expire(key, seconds);
            }
        }
    }
}
