package com.bxm.lovelink.common.push.factory.impl;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.lovelink.common.contant.Constants;
import com.bxm.lovelink.common.dal.entity.Activity;
import com.bxm.lovelink.common.dal.entity.ActivityRegistration;
import com.bxm.lovelink.common.dal.entity.PushTask;
import com.bxm.lovelink.common.dal.service.IActivityRegistrationService;
import com.bxm.lovelink.common.dal.service.IActivityService;
import com.bxm.lovelink.common.dal.service.IPushTaskService;
import com.bxm.lovelink.common.push.TaskRedisKeys;
import com.bxm.lovelink.common.push.factory.AbstractTimerPushSceneService;
import com.bxm.lovelink.common.push.factory.Push;
import com.bxm.lovelink.common.push.factory.PushScene;
import com.bxm.lovelink.common.push.factory.PushTarget;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 活动开始推送
 * @author: weixing
 */
@Slf4j
@Service
public class ActivityStartTimerPushSceneService extends AbstractTimerPushSceneService {
    private final IActivityService activityService;
    private final IActivityRegistrationService activityRegistrationService;
    private final Fetcher fetcher;


    protected ActivityStartTimerPushSceneService(IPushTaskService pushTaskService, Fetcher fetcher, IActivityService activityService, IActivityRegistrationService activityRegistrationService) {
        super(pushTaskService, fetcher);
        this.activityService = activityService;
        this.activityRegistrationService = activityRegistrationService;
        this.fetcher = fetcher;
    }

    @Override
    protected List<PushTarget> findPushTaskTargets(PushTask pushTask) {
        String taskSceneCustomConfig = pushTask.getTaskSceneCustomConfig();
        int hours = NumberUtils.toInt(taskSceneCustomConfig);
        if (hours <= 0) {
            return null;
        }
        // 查找离开始时间不足hours的活动
        LambdaQueryWrapper<Activity> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.eq(Activity::getActivityStatus, Constants.ActivityStatus.OPEN);
        lambdaQueryWrapper
                .gt(Activity::getActivityStartTime, LocalDateTime.now())
                .le(Activity::getActivityStartTime, LocalDateTime.now().plusHours(hours));
        List<Activity> activityList = activityService.list(lambdaQueryWrapper);
        List<Long> activityIds = activityList.stream().filter(Objects::nonNull).map(Activity::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(activityIds)){
            return Collections.emptyList();
        }
        List<ActivityRegistration> activityRegistrations = activityRegistrationService.list(new LambdaQueryWrapper<ActivityRegistration>().in(ActivityRegistration::getActivityId, activityIds).eq(ActivityRegistration::getStatus, Constants.ActivityRegistrationStatus.SUCCESS));
        if (CollectionUtils.isEmpty(activityRegistrations)) {
            return Collections.emptyList();
        }
        List<ActivityRegistration> activityRegistrationList = new ArrayList<>();
        for (ActivityRegistration activityRegistration : activityRegistrations) {
            String targetId = activityRegistration.getActivityId() + ":" + activityRegistration.getUserId();
            KeyGenerator counterKey = TaskRedisKeys.stringPushCount(pushTask.getId(), targetId);
            Long currentCount = Optional.ofNullable(fetcher.fetch(counterKey, Long.class)).orElse(0L);
            if (currentCount >= pushTask.getPushMaximumCount()) {
                continue;
            }
            activityRegistrationList.add(activityRegistration);
        }
        return activityRegistrationList.stream().map(activityRegistration -> new PushTarget()
                .setUserId(activityRegistration.getUserId())
                .addExtend("activityId", activityRegistration.getActivityId())).collect(Collectors.toList());
    }

    @Override
    public Map<String, Object> createTemplateParameters(PushTask pushTask, PushTarget pushTarget) {
        Long activityId = (Long) pushTarget.getExtend().get("activityId");
        Activity activity = activityService.getById(activityId);
        // <活动名称>还有<hour>开始，请安排时间及时参加哦
        return new HashMap<String, Object>() {{
            put("hour", pushTask.getTaskSceneCustomConfig());
            put("activityName", activity.getName());
        }};
    }

    @Override
    public PushScene of() {
        return PushScene.ACTIVITY_START;
    }

    @Override
    public void onAfterPostEvent(List<Push> pushList) {
        for (Push push : pushList) {
            PushTarget pushTarget = push.getPushTarget();
            Long activityId = (Long) pushTarget.getExtend().get("activityId");
            Long userId = pushTarget.getUserId();
            PushTask pushTask = push.getPushTask();
            String targetId = activityId + ":" + userId;
            KeyGenerator counterKey = TaskRedisKeys.stringPushCount(pushTask.getId(), targetId);
            String taskSceneCustomConfig = pushTask.getTaskSceneCustomConfig();
            int hours = NumberUtils.toInt(taskSceneCustomConfig);
            increment(counterKey, hours);
        }
    }

    @Override
    public Boolean isCheckIfPushOnConditional() {
       return false;
    }
}
