package com.bxm.component.mq.listener.rabbit;

import com.bxm.component.mq.config.RabbitConfigurationProperties;
import com.bxm.component.mq.consumer.MessageConsumer;
import com.bxm.component.mq.enums.MessageTypeEnum;
import com.bxm.component.mq.conditionals.MessageConditionalOnProperty;
import com.bxm.newidea.component.tools.ReflectionUtils;
import com.rabbitmq.client.Channel;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.connection.Connection;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Objects;

/**
 * 远程事件监听器
 * 接收到远程事件后，根据事件中的值类型进行对应的分发处理
 *
 * @author lowi
 * @date 2021年8月18日16:13:40
 **/
@Component
@Slf4j
@AllArgsConstructor
@MessageConditionalOnProperty(type = MessageTypeEnum.RABBIT_MQ)
public class RabbitMqEventListener implements ApplicationRunner, ApplicationContextAware {

    private final ConnectionFactory connectionFactory;

    private final RabbitConfigurationProperties configurationProperties;

    private ApplicationContext applicationContext;

    @Override
    public void run(ApplicationArguments args) {
        Map<String, MessageConsumer> subscriberMap = applicationContext.getBeansOfType(MessageConsumer.class);

        for (MessageConsumer subscriber : subscriberMap.values()) {
            Class<?> subscriberValueClass = ReflectionUtils.getFirstGenericType(subscriber.getClass());
            if (null == subscriberValueClass) {
                log.error("声明的[{}]没有指定泛型，无法匹配订阅数据", subscriber.getClass().getSimpleName());
                continue;
            }
            Method method = null;
            try {
                method = subscriber.getClass().getMethod("uniqueMark");
                Object invoke = method.invoke(subscriber.getClass().newInstance());
                if (Objects.isNull(invoke)) {
                    log.error("声明的[{}]===========没有指定泛型，无法匹配订阅数据", subscriber);
                    return;
                }
                getQueueChannel((String) invoke, subscriber, subscriberValueClass);
            } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException | InstantiationException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 消息消费设置
     *
     * @param queue      队列名
     * @param subscriber 实现类
     * @param className  具体的实现类的bean
     */
    private void getQueueChannel(String queue, MessageConsumer subscriber, Class<?> className) {
        Connection connection = connectionFactory.createConnection();
        Channel channel = connection.createChannel(false);
        String exchange = queue + "-exchange";
        String routingKey = queue + "-routing";
        try {

            channel.exchangeDeclare(exchange, configurationProperties.getType(), configurationProperties.getDurable(),
                    configurationProperties.getAutoDelete(), configurationProperties.getArguments());
            channel.queueDeclare(queue, configurationProperties.getDurable(), configurationProperties.getExclusive(),
                    configurationProperties.getAutoDelete(), configurationProperties.getArguments());
            // 将队列绑定到交换机
            //
            // param1：destination，目的地，队列的名字
            // param2：source，资源，交换机的名字
            // param3：routingKey，路由键（目前没有用到routingKey，填 "" 即可）
            channel.queueBind(queue, exchange, routingKey);
            //发送消息
            channel.basicConsume(queue, true, new DefaultRabbitConsumer(channel, subscriber, className));
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
