package com.bxm.component.bus.config;

import com.bxm.component.bus.event.LogSubscriberExceptionHandler;
import com.bxm.component.bus.guava.AsyncEventBus;
import com.bxm.component.bus.guava.EventBus;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.MoreExecutors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import java.util.Map;
import java.util.concurrent.*;

/**
 * 消息总线配置
 *
 * @author liujia
 * @date 7/1/21 5:12 PM
 **/
@Slf4j
@Configuration
@EnableConfigurationProperties(ComponentBusConfigurationProperties.class)
@Import(EventBusImportBeanDefinitionRegistrar.class)
public class ComponentBusConfiguration {

    private ComponentBusConfigurationProperties properties;

    /**
     * 异步事件总线对应的执行器，可根据情况选择性的替换
     */
    public static final String COMPONENT_ASYNC_EXECUTOR = "COMPONENT_ASYNC_EXECUTOR";

    private Map<String, RejectedExecutionHandler> handlerMap;

    public ComponentBusConfiguration(ComponentBusConfigurationProperties properties) {
        this.properties = properties;

        handlerMap = Maps.newHashMap();
        handlerMap.put(
            ThreadPoolExecutor.CallerRunsPolicy.class.getSimpleName(),
            new ThreadPoolExecutor.CallerRunsPolicy()
        );
        handlerMap.put(
            ThreadPoolExecutor.AbortPolicy.class.getSimpleName(),
            new ThreadPoolExecutor.AbortPolicy()
        );
        handlerMap.put(
            ThreadPoolExecutor.DiscardPolicy.class.getSimpleName(),
            new ThreadPoolExecutor.DiscardPolicy()
        );
        handlerMap.put(
            ThreadPoolExecutor.DiscardOldestPolicy.class.getSimpleName(),
            new ThreadPoolExecutor.DiscardOldestPolicy()
        );
    }

    /**
     * 默认提供异步事务总线
     *
     * @param executor 线程执行器
     * @return 异步事件总线
     */
    @Bean(name = {
        "asyncEventBus",
        "eventBus"
    })
    public EventBus asyncEventBus(@Qualifier(COMPONENT_ASYNC_EXECUTOR) Executor executor) {
        AsyncEventBus asyncEventBus = new AsyncEventBus(
            "component-async-event-bus",
            executor,
            new LogSubscriberExceptionHandler()
        );
        ComponentEventBus.asyncEventBus = asyncEventBus;
        return asyncEventBus;
    }

    @Bean(COMPONENT_ASYNC_EXECUTOR)
    @ConditionalOnMissingBean(name = COMPONENT_ASYNC_EXECUTOR)
    public Executor asyncEventBusExecutor() {
        LinkedBlockingQueue<Runnable> blockingQueue = new LinkedBlockingQueue<>();

        ThreadPoolExecutor executor = new ThreadPoolExecutor(
            properties.getCorePoolSize(),
            properties.getCorePoolSize(),
            0,
            TimeUnit.MILLISECONDS,
            blockingQueue,
            new NamedThreadFactory("component-async-event-bus")
        );

        log.info(
            "初始化异步时间处理线程池，核心线程数：{}，最大线程数：{},拒绝策略：{}",
            properties.getCorePoolSize(),
            properties.getCorePoolSize(),
            properties.getExecutorRejectHandlerPolicy()
        );

        executor.allowCoreThreadTimeOut(false);
        executor.setRejectedExecutionHandler(new RejectedExecutionHandler() {
            @Override
            public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {

                if (!executor.isTerminating() && !executor.isTerminated()) {
                    log.error(
                        "超出线程池处理能力,当前拒绝策略：{},当前线程执行器状态：{}",
                        properties.getExecutorRejectHandlerPolicy(),
                        executor.toString()
                    );

                    getHandler(properties.getExecutorRejectHandlerPolicy()).rejectedExecution(r, executor);
                } else {
                    log.info("线程终止，处理剩余任务，线程执行器状态：{}", executor.toString());
                    getHandler(ThreadPoolExecutor.CallerRunsPolicy.class.getSimpleName())
                        .rejectedExecution(r, executor);
                }
            }
        });

        // 关闭服务时，等待处理完成
        return MoreExecutors.getExitingExecutorService(executor);
    }

    private RejectedExecutionHandler getHandler(String policy) {
        return handlerMap.getOrDefault(
            policy,
            new ThreadPoolExecutor.AbortPolicy()
        );
    }
}
























