package com.bxm.newidea.component.tools;

import com.bxm.newidea.component.tools.inner.ValidateMessages;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.BindException;
import org.springframework.validation.DirectFieldBindingResult;
import org.springframework.validation.FieldError;

import java.util.HashMap;
import java.util.Map;

/**
 * 校验构造器，用于进行单个属性的校验
 */
public class ValidBuilder {

    private boolean direct;

    public ValidBuilder(boolean direct) {
        this.direct = direct;
        bindingResult = new DirectFieldBindingResult(this, this.getClass().getSimpleName());
    }

    private Map<Object, Object> validMap = new HashMap<>();

    private DirectFieldBindingResult bindingResult;

    private String field;

    private Object value;

    /**
     * 添加需要校验的属性
     *
     * @param field 属性名称
     * @param value 属性值
     * @return 校验构造器
     */
    public ValidBuilder prop(String field, Object value) {
        this.field = field;
        this.value = value;
        return this;
    }

    /**
     * @see #notBlank(String)
     */
    public ValidBuilder notBlank() throws BindException {
        return notBlank(null);
    }

    /**
     * 验证属性值不能为空
     *
     * @param message 验证错误时的返回消息，如果为null或者空则返回对应的默认消息
     * @return 校验构造器
     * @throws BindException 如果是direct模式，直接抛出异常，反之则需要手动调用valid()
     */
    public ValidBuilder notBlank(String message) throws BindException {
        if (null == value || StringUtils.isBlank(value.toString())) {
            addError(message, ValidateMessages.NOT_BLANK);
        }
        return this;
    }

    /**
     * @see #max(long, String)
     */
    public ValidBuilder max(long size) throws BindException {
        return max(size,null);
    }

    /**
     * 数字：  验证属性值不能大于size（不包含等于）
     * 字符串：验证属性长度不能大于size（不包含等于）
     * @param size 最大的数字或最长的长度
     * @param message 验证失败的返回消息
     * @return 校验构造器
     * @throws BindException 如果是direct模式，直接抛出异常，反之则需要手动调用valid()
     */
    public ValidBuilder max(long size, String message) throws BindException {
        if (null != value) {
            //获取基础类型，并转化为对应的类型
            if (ReflectionUtils.isNumeric(value)) {
                long numValue = NumberUtils.parseToLong(value.toString());
                if (numValue > size) {
                    addError(message, ValidateMessages.MAX_NUM, size);
                }
            }else{
                if(value.toString().length() > size){
                    addError(message, ValidateMessages.MAX_SIZE, size);
                }
            }
        }
        return this;
    }

    /**
     * @see #min(long, String)
     */
    public ValidBuilder min(long size) throws BindException {
        return min(size,null);
    }

    /**
     * 数字：  验证属性值不能小于size（不包含等于）
     * 字符串：验证属性长度不能小于size（不包含等于）
     * @param size 最大的数字或最长的长度
     * @param message 验证失败的返回消息
     * @return 校验构造器
     * @throws BindException 如果是direct模式，直接抛出异常，反之则需要手动调用valid()
     */
    public ValidBuilder min(long size, String message) throws BindException {
        if (null != value) {
            //获取基础类型，并转化为对应的类型
            if (ReflectionUtils.isNumeric(value)) {
                long numValue = NumberUtils.parseToLong(value.toString());
                if (numValue < size) {
                    addError(message, ValidateMessages.MIN_NUM, size);
                }
            }else{
                if(value.toString().length() < size){
                    addError(message, ValidateMessages.MIN_SIZE, size);
                }
            }
        }
        return this;
    }

    private void addError(String message, String defaultMsg, Object... replaces) throws BindException {
        if (StringUtils.isBlank(message)) {
            message = defaultMsg;
            if (null != replaces) {
                for (Object obj : replaces) {
                    message = StringUtils.replace(message, "{}", obj.toString());
                }
            }
        }
        bindingResult.addError(new FieldError(this.getClass().getSimpleName(), field, message));
        isDirect();
    }

    /**
     * 是否需要直接抛出校验异常
     *
     * @throws BindException 校验异常信息
     */
    private void isDirect() throws BindException {
        if (direct) {
            valid();
        }
    }

    /**
     * 进行最终验证
     *
     * @throws BindException 验证是否成功
     */
    public void valid() throws BindException {
        if (bindingResult.hasErrors()) {
            throw new BindException(bindingResult);
        }
    }
}