package com.bxm.newidea.component.tools;

import com.bxm.newidea.component.tools.inner.IntervalMap;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.math.NumberUtils;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.*;

/**
 * 日期操作工具类
 *
 * @author liujia 2018/3/28 18:09
 */
@SuppressWarnings({
    "WeakerAccess",
    "unused"
})
@UtilityClass
@Slf4j
public class DateUtils {

    /**
     * 一周的毫秒数
     */
    public static final long WEEK_MILLISECOND = 7 * 24 * 60 * 60 * 1000L;

    /**
     * 一天的毫秒数
     */
    public static final long DAY_MILLISECOND = 24 * 60 * 60 * 1000L;

    /**
     * 一小时的毫秒数
     */
    public static final long HOUR_MILLISECOND = 60 * 60 * 1000L;

    /**
     * 一分钟的毫秒数
     */
    public static final long MINUTE_MILLISECOND = 60 * 1000L;

    public static final String DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

    public static final String DATE_TIME_COMPACT_FORMAT = "yyyyMMddHHmmss";

    public final static String DATE_FORMAT = "yyyy-MM-dd";

    public final static String DATE_COMPACT_FORMAT = "yyyyMMdd";

    public final static String ISO_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";

    public final static String DATE_WITH_HOUR_FORMAT = "yyyy-MM-dd HH";

    public final static String DATE_WITHOUT_DAY_FORMAT = "yyyyMM";

    public final static String DATE_FORMAT_DAY_HOUR_MUNITE = "yyyy-MM-dd HH:mm";

    public final static String DATE_FORMAT_YEAR_MONTH_DAY_FORMAT = "yyyy年MM月dd日";

    public final static String DATE_FORMAT_MONTH_DAY_FORMAT = "MM月dd日";

    public final static String DATE_TIME_SECONDMS_FORMAT = "yyyyMMddHHmmsssss";

    public final static String MONTH_DAY_FORMAT = "MM-dd";

    public final static String HOUR_MINUTE_FORMAT = "HH:mm";

    public final static String HOUR_MINUTE_SECOND_FORMAT = "HH:mm:ss";

    public static ThreadLocal<DateFormat> DATE_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_FORMAT));

    public static ThreadLocal<DateFormat> DATE_FORMAT_MONTH_DAY_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_FORMAT_MONTH_DAY_FORMAT));

    public static ThreadLocal<DateFormat> DATE_TIME_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_TIME_FORMAT));

    public static ThreadLocal<DateFormat> ISO_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(ISO_FORMAT));

    public static ThreadLocal<DateFormat> DATE_HOUR_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_WITH_HOUR_FORMAT));

    public static ThreadLocal<DateFormat> DATE_TIME_SECOND_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_TIME_COMPACT_FORMAT));

    public static ThreadLocal<DateFormat> DATE_TIME_SECONDMS_FORMAT_THREAD_LOCAL = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_TIME_SECONDMS_FORMAT));

    public static ThreadLocal<DateFormat> PATTERN_NO_DELIMITER_WITHOUT_DAY_FORMAT = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_WITHOUT_DAY_FORMAT));

    public static ThreadLocal<DateFormat> PATTERN_NO_DELIMITER_FORMAT = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_COMPACT_FORMAT));

    public static ThreadLocal<DateFormat> PATTERN_MONTH_DAY_FORMAT = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(MONTH_DAY_FORMAT));

    public static ThreadLocal<DateFormat> PATTERN_YEAR_MONTH_DAY_FORMAT = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(DATE_FORMAT_YEAR_MONTH_DAY_FORMAT));

    public static ThreadLocal<DateFormat> PATTERN_HOUR_MINUTE_FORMAT = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(HOUR_MINUTE_FORMAT));

    public static ThreadLocal<DateFormat> PATTERN_HOUR_MINUTE_SECEND_FORMAT = ThreadLocal.withInitial(
        () -> new SimpleDateFormat(HOUR_MINUTE_SECOND_FORMAT));

    /**
     * 解析列表，根据配置的解析格式进行尝试解析，如果能解析则直接返回
     */
    private static List<ThreadLocal<DateFormat>> FORMAT_LIST = new ArrayList<>();

    static {
        FORMAT_LIST.add(DATE_TIME_FORMAT_THREAD_LOCAL);
        FORMAT_LIST.add(DATE_FORMAT_THREAD_LOCAL);
        FORMAT_LIST.add(ISO_FORMAT_THREAD_LOCAL);
        FORMAT_LIST.add(DATE_TIME_SECONDMS_FORMAT_THREAD_LOCAL);
        FORMAT_LIST.add(DATE_TIME_SECOND_FORMAT_THREAD_LOCAL);
        FORMAT_LIST.add(DATE_HOUR_FORMAT_THREAD_LOCAL);
    }

    /**
     * 将日期格式化为日期字符串
     *
     * @param date 日期,如果参数为null将返回null
     * @return 日期类型的字符串，不包含时间
     */
    public static String formatDate(Date date) {
        if (null == date) {
            return null;
        }
        return DATE_FORMAT_THREAD_LOCAL.get().format(date);
    }

    /**
     * 根据设定的任意格式进行日期格式化
     *
     * @param source       原始日期类型
     * @param formatPatter 格式化字符串
     * @return 格式化结果，如果source为null则返回空字符串
     */
    public static String formatAtWill(Date source, String formatPatter) {
        if (null == source) {
            return StringUtils.EMPTY;
        }
        SimpleDateFormat format = new SimpleDateFormat(formatPatter);
        return format.format(source);
    }

    /**
     * 根据支持的所有日期格式将日期转换为字符串
     *
     * @param date 日期，如果为null则返回null
     * @return 设定格式的字符串
     */
    public static String formatDateNonStrict(Date date) {
        String result = null;
        if (null == date) {
            return null;
        }

        for (ThreadLocal<DateFormat> supportFormatLocal : FORMAT_LIST) {
            result = supportFormatLocal.get().format(date);

            if (StringUtils.isNotBlank(result)) {
                return result;
            }
        }

        return result;
    }

    /**
     * 将日期格式化为时间戳字符串
     *
     * @param date 日期,如果参数为null将返回null
     * @return 日期类型的字符串，包含时间
     */
    public static String formatDateTime(Date date) {
        if (null == date) {
            return null;
        }
        return DATE_TIME_FORMAT_THREAD_LOCAL.get().format(date);
    }

    /**
     * 将字符串解析为日期对象
     *
     * @param source 源字符串,源字符串为空时返回null
     * @return 日期对象
     */
    public static Date parseDate(String source) {
        if (StringUtils.isBlank(source)) {
            return null;
        }

        try {
            return DATE_FORMAT_THREAD_LOCAL.get().parse(source);
        } catch (ParseException e) {
            log.info(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 采用不严格的格式进行日期格式化，会尝试所有注册的解析格式，如果匹配则进行返回，最后还会尝试unix时间戳的方式
     * 注意：注册的解析格式严格的在线，不然会出现精度丢失的问题
     *
     * @param source 原始字符串，如果为空则返回null
     * @return 解析后的日期，如果无法解析则返回null
     */
    public static Date parseDateNonStrict(String source) {
        if (null == source) {
            return null;
        }

        if (NumberUtils.isDigits(source) && source.length() == 13) {
            return new Date(Long.valueOf(source));
        }

        Date result;
        for (ThreadLocal<DateFormat> supportFormatLocal : FORMAT_LIST) {
            try {
                result = supportFormatLocal.get().parse(source);
                if (null != result) {
                    return result;
                }
            } catch (ParseException e) {
                log.info(e.getMessage(), e);
            }
        }

        return null;
    }

    /**
     * 将字符串解析为日期对象
     * 日期格式为：yyyy-MM-dd HH:mm:ss
     *
     * @param source 源字符串,源字符串为空时返回null
     * @return 日期对象
     */
    public static Date parseDateTime(String source) {
        if (StringUtils.isBlank(source)) {
            return null;
        }

        try {
            return DATE_TIME_FORMAT_THREAD_LOCAL.get().parse(source);
        } catch (ParseException e) {
            log.info(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 将源字符串解析为日期对象，根据支持的日期格式进行解析，如日期类型不可解析时返回null<br/>
     * 格式不明确的场景调用此方法，格式明确时调用{@link #parseDateTime(String)}或{@link #parseDate(String)}
     *
     * @param source 源字符串,源字符串为空时返回null
     * @return 日期对象
     */
    public static Date parse(String source) {
        if (StringUtils.isBlank(source)) {
            return null;
        }

        Date parseResult;

        try {
            parseResult = DATE_TIME_FORMAT_THREAD_LOCAL.get().parse(source);
        } catch (ParseException e) {
            parseResult = null;
        }
        try {
            if (null == parseResult) {
                parseResult = DATE_FORMAT_THREAD_LOCAL.get().parse(source);
            }
        } catch (ParseException e) {
            log.info(e.getMessage(), e);
        }

        if (null == parseResult) {
            log.info("不支持该格式的日期转换，源数据为[{}]", source);
        }

        return parseResult;
    }

    /**
     * 获取当前日期的日期格式字符串
     *
     * @return 日期格式字符串
     */
    public static String getCurrentDate() {
        return formatDate(new Date());
    }

    /**
     * 获取当前时间的时间格式字符串
     *
     * @return 时间格式字符串
     */
    public static String getCurrentDateTime() {
        return formatDateTime(new Date());
    }

    /**
     * 获取去除了时分秒毫秒的日期
     *
     * @param date 日期,如果传递null则会返回当前日期
     * @return 格式化后的日期
     */
    public static Date getClearDate(Date date) {
        if (null == date) {
            date = new Date();
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        clearTimePart(calendar);
        return calendar.getTime();
    }

    /**
     * 指定时间是否在当前时间之前
     *
     * @param when 时间点
     * @return true表示参数时间在当前时间之前
     */
    public static boolean before(Date when) {
        return after(new Date(), when);
    }

    /**
     * 指定时间是否在当前时间之后
     *
     * @param when 时间点
     * @return true表示参数时间在当前时间之后
     */
    public static boolean after(Date when) {
        return after(when, new Date());
    }

    /**
     * 比较两个日期，判断<code>source</code>是否在<code>target</code>之前
     *
     * @param source 源时间
     * @param target 比较的目标时间
     * @return true表示source在target之前，false反之
     */
    public static boolean before(Date source, Date target) {
        return after(target, source);
    }

    /**
     * 比较两个日期，判断<code>source</code>是否在<code>target</code>之后
     *
     * @param source 源时间
     * @param target 比较的目标时间
     * @return true表示source在target之后，false反之
     */
    public static boolean after(Date source, Date target) {
        if (null == source || null == target) {
            return false;
        }
        Calendar sourceCalendar = Calendar.getInstance();
        sourceCalendar.setTime(source);

        return source.after(target);
    }

    /**
     * 日期中添加对应字段的值，支持负数
     *
     * @param date   原始日期
     * @param field  修改的字符类型，取值见{@link Calendar}
     * @param amount 修改的值，支持负数
     * @return 修改后的日期
     */
    public static Date addField(Date date, int field, int amount) {
        Calendar sourceCalendar = Calendar.getInstance();
        sourceCalendar.setTime(date);
        sourceCalendar.add(field, amount);
        return sourceCalendar.getTime();
    }

    /**
     * 日期中获取对应字段的值
     *
     * @param date  原始日期
     * @param field 修改的字符类型，取值见{@link Calendar}
     * @return 修改后的日期
     */
    public static int getField(Date date, int field) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        return calendar.get(field);
    }

    /**
     * 日期中设置对应字段的值，支持负数
     *
     * @param date   原始日期
     * @param field  设置的字符类型，取值见{@link Calendar}
     * @param amount 设置的值
     * @return 设置后的日期
     */
    public static Date setField(Date date, int field, int amount) {
        Calendar sourceCalendar = Calendar.getInstance();
        sourceCalendar.setTime(date);
        sourceCalendar.set(field, amount);
        return sourceCalendar.getTime();
    }

    /**
     * 计算2个日期值之间相差的月份，不论天数
     *
     * @param source 起始日期,不能为空，否则返回0
     * @param target 截止日期,不能为空，否则返回0
     * @return 相差月份
     */
    public static int getDiffMonths(Date source, Date target) {
        if (null == source || null == target) {
            return 0;
        }
        Calendar sourceCalendar = Calendar.getInstance();
        sourceCalendar.setTime(source);
        Calendar targetCalendar = Calendar.getInstance();
        targetCalendar.setTime(target);

        int sourceYear = sourceCalendar.get(Calendar.YEAR);
        int sourceMonth = sourceCalendar.get(Calendar.MONTH);

        int targetYear = targetCalendar.get(Calendar.YEAR);
        int targetMonth = targetCalendar.get(Calendar.MONTH);

        int diffMonth = (sourceYear * 12 + sourceMonth) - (targetYear * 12 + targetMonth);
        return Math.abs(diffMonth);
    }

    /**
     * 计算2个日期值之间相差的天数
     *
     * @param source 起始日期,不能为空，否则返回0
     * @param target 截止日期,不能为空，否则返回0
     * @param abs    true表示返回绝对值
     * @return 相差天数
     */
    public static int getDiffDays(Date source, Date target, boolean abs) {
        if (null == source || null == target) {
            return 0;
        }

        LocalDate sourceDate = LocalDateUtils.toLocalDate(source);
        LocalDate targetDate = LocalDateUtils.toLocalDate(target);

        long diffDays = sourceDate.toEpochDay() - targetDate.toEpochDay();

        if (abs) {
            return (int) Math.abs(diffDays);
        }
        return (int) diffDays;
    }

    /**
     * 计算2个日期之间相差的小时数
     *
     * @param source 起始日期,不能为空，否则返回0
     * @param target 截止日期,不能为空，否则返回0
     * @param abs    true表示返回绝对值
     * @return 相差小时
     */
    public static long getDiffHours(Date source, Date target, boolean abs) {
        if (null == source || null == target) {
            return 0;
        }
        long diffSecond = source.getTime() - target.getTime();
        if (abs) {
            diffSecond = Math.abs(diffSecond);
        }
        return Math.abs((diffSecond / HOUR_MILLISECOND));
    }

    /**
     * 计算2个日期相差的分钟数
     *
     * @param source 起始日期,不能为空，否则返回0
     * @param target 截止日期,不能为空，否则返回0
     * @return 相差分钟数的绝对值
     */
    public static long getDiffMinutes(Date source, Date target) {
        return getDiffMinutes(source, target, true);
    }

    /**
     * 计算2个日期相差的分钟数
     *
     * @param source 起始日期,不能为空，否则返回0
     * @param target 截止日期,不能为空，否则返回0
     * @param abs    true表示返回绝对值
     * @return 相差的分钟数
     */
    public static long getDiffMinutes(Date source, Date target, boolean abs) {
        if (null == source || null == target) {
            return 0;
        }
        long diffSeconed = source.getTime() - target.getTime();
        if (abs) {
            diffSeconed = Math.abs(diffSeconed);
        }
        return (diffSeconed / MINUTE_MILLISECOND);
    }

    /**
     * 计算两个日期相差的秒数
     *
     * @param source 起始日期
     * @param target 截止日期
     * @param abs    是否返回绝对值
     * @return 相差的秒数，如果任意参数为空返回0
     */
    public static long getDiffSeconed(Date source, Date target, boolean abs) {
        if (null == source || null == target) {
            return 0;
        }
        long diffSecond = (source.getTime() - target.getTime()) / 1000;
        if (abs) {
            return Math.abs(diffSecond);
        }
        return diffSecond;
    }

    /**
     * 获取2个日期之间相差的年，一般用于计算年年
     *
     * @param source 开始日期
     * @param target 结束日期
     * @return 相差的年数
     */
    public static int getDiffYears(Date source, Date target) {
        if (null == source || null == target) {
            return 0;
        }
        Calendar sourceCalendar = Calendar.getInstance();
        sourceCalendar.setTime(source);

        Calendar targetCalendar = Calendar.getInstance();
        targetCalendar.setTime(target);

        int diffYears = targetCalendar.get(Calendar.YEAR) - sourceCalendar.get(Calendar.YEAR);
        diffYears = Math.abs(diffYears);

        if (targetCalendar.get(Calendar.MONTH) < sourceCalendar.get(Calendar.MONTH)) {
            diffYears--;
        } else if (targetCalendar.get(Calendar.MONTH) == sourceCalendar.get(Calendar.MONTH)) {
            if (targetCalendar.get(Calendar.DAY_OF_MONTH) < sourceCalendar.get(Calendar.DAY_OF_MONTH)) {
                diffYears--;
            }
        }
        return diffYears;
    }

    /**
     * 获取当前时间对于的日期区间，具体支持的区间可查看{@link Interval}
     *
     * @param interval {@link Interval}
     * @return 区间映射，包含区间的起止时间
     */
    public static IntervalMap getInterval(Interval interval) {
        return getInterval(interval, new Date());
    }

    /**
     * 获取指定日期对应的日期区间，具体指出的区间可查看{@link Interval}
     *
     * @param interval {@link Interval}
     * @param date     指定的检索日期
     * @return 区间映射，包含区间的起止时间
     */
    public static IntervalMap getInterval(Interval interval, Date date) {
        Date startTime = null;
        Date endTime = null;

        if (null == interval || null == date) {
            return new IntervalMap(null, null);
        }

        Calendar startCalendar = Calendar.getInstance();
        Calendar endCalendar = Calendar.getInstance();

        int currentMonth, startMonth, endMonth;

        switch (interval) {
            case TODAY:
                startCalendar.setTime(date);

                endCalendar.setTime(date);
                endCalendar.add(Calendar.DAY_OF_YEAR, 1);

                break;
            case TOMORROW:
                startCalendar.setTime(date);
                startCalendar.add(Calendar.DAY_OF_YEAR, 1);

                endCalendar.setTime(date);
                endCalendar.add(Calendar.DAY_OF_YEAR, 2);
                break;
            case YESTERDAY:
                startCalendar.setTime(date);
                startCalendar.add(Calendar.DAY_OF_YEAR, -1);

                endCalendar.setTime(date);

                break;
            case THREE_DAY:
                startCalendar.setTime(date);
                startCalendar.add(Calendar.DAY_OF_YEAR, -3);

                endCalendar.setTime(date);

                break;
            case FIVE_DAY:
                startCalendar.setTime(date);
                startCalendar.add(Calendar.DAY_OF_YEAR, -5);

                endCalendar.setTime(date);

                break;
            case SEVEN_DAY:
                startCalendar.setTime(date);
                startCalendar.add(Calendar.DAY_OF_YEAR, -7);

                endCalendar.setTime(date);

                break;
            case WEEK:
                startCalendar.setTime(date);

                boolean isSunday = startCalendar.get(Calendar.DAY_OF_WEEK) == Calendar.SUNDAY;
                if (isSunday) {
                    startCalendar.add(Calendar.WEEK_OF_YEAR, -1);
                }
                startCalendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);

                endCalendar.setTime(date);
                if (!isSunday) {
                    endCalendar.add(Calendar.WEEK_OF_YEAR, 1);
                }
                endCalendar.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);

                break;
            case MONTH:
                startCalendar.setTime(date);
                startCalendar.set(Calendar.DAY_OF_MONTH, 1);

                endCalendar.setTime(date);
                endCalendar.set(Calendar.DAY_OF_MONTH, endCalendar.getActualMaximum(Calendar.DAY_OF_MONTH));

                break;
            case QUARTER:
                startCalendar.setTime(date);
                endCalendar.setTime(date);

                currentMonth = startCalendar.get(Calendar.MONTH);

                startMonth = (currentMonth % 3) * 3;
                endMonth = startMonth + 2;

                startCalendar.set(Calendar.MONTH, startMonth);
                startCalendar.set(Calendar.DAY_OF_MONTH, 1);

                endCalendar.set(Calendar.MONTH, endMonth);
                endCalendar.set(Calendar.DAY_OF_MONTH, endCalendar.getActualMaximum(Calendar.DAY_OF_MONTH));

                break;
            case HALY_YEAR:
                startCalendar.setTime(date);
                endCalendar.setTime(date);

                currentMonth = startCalendar.get(Calendar.MONTH);

                startMonth = (currentMonth % 6) * 6;
                endMonth = startMonth + 5;

                startCalendar.set(Calendar.MONTH, startMonth);
                startCalendar.set(Calendar.DAY_OF_MONTH, 1);

                endCalendar.set(Calendar.MONTH, endMonth);
                endCalendar.set(Calendar.DAY_OF_MONTH, endCalendar.getActualMaximum(Calendar.DAY_OF_MONTH));
                break;
            case YEAR:
                startCalendar.setTime(date);
                startCalendar.set(Calendar.MONTH, 0);
                startCalendar.set(Calendar.DAY_OF_MONTH, 1);

                endCalendar.setTime(date);
                endCalendar.set(Calendar.MONTH, 11);
                endCalendar.set(Calendar.DAY_OF_MONTH, endCalendar.getActualMaximum(Calendar.DAY_OF_MONTH));
                break;
            case LAST_YEAR:
                startCalendar.setTime(date);
                startCalendar.add(Calendar.YEAR, -1);
                startCalendar.set(Calendar.MONTH, 0);
                startCalendar.set(Calendar.DAY_OF_MONTH, 1);

                endCalendar.setTime(date);
                endCalendar.add(Calendar.YEAR, -1);
                endCalendar.set(Calendar.MONTH, 11);
                endCalendar.set(Calendar.DAY_OF_MONTH, endCalendar.getActualMaximum(Calendar.DAY_OF_MONTH));
                break;
            default:
                return new IntervalMap(startTime, endTime);
        }

        clearTimePart(startCalendar);
        clearTimePart(endCalendar);

        startTime = startCalendar.getTime();
        endTime = endCalendar.getTime();

        return new IntervalMap(startTime, endTime);
    }

    /**
     * 清理日期中的时分秒毫秒部分
     */
    public static void clearTimePart(Calendar calendar) {
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
    }

    /**
     * 清理日期中的时分秒毫秒部分
     */
    public static Date clearTimePart(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        clearTimePart(calendar);
        return calendar.getTime();
    }

    /**
     * 获取当前时间到今天结束之间的毫秒值
     *
     * @return 距离当日23:59:59还剩余多少毫秒
     */
    public static long getTodayEndMillis() {
        Date now = new Date();

        return getDayEndTime(null).getTime() - now.getTime();
    }

    /**
     * 获取当天剩余秒数
     *
     * @return 距离当日23:59:59还剩余多少秒
     */
    public static int getCurSeconds() {
        return (int) (getTodayEndMillis() / 1000);
    }

    /**
     * 获取日期的最后时刻
     *
     * @param date 需要处理的日期
     * @return 处理后的日期，时分秒为：23:59:59
     */
    public static Date getDayEndTime(Date date) {
        Calendar instance = Calendar.getInstance();

        if (Objects.nonNull(date)) {
            instance.setTime(date);
        }

        instance.set(Calendar.HOUR_OF_DAY, 23);
        instance.set(Calendar.MINUTE, 59);
        instance.set(Calendar.SECOND, 59);
        instance.set(Calendar.MILLISECOND, 999);

        return instance.getTime();
    }

    /**
     * 判断两个日期是否是同一天
     *
     * @param resource 来源日期
     * @param target   目标日期
     * @return 是同一天-true,不是同一天-false
     */
    public static boolean isSameDay(Date resource, Date target) {
        if (resource != null && target != null) {
            Calendar cal1 = Calendar.getInstance();
            cal1.setTime(resource);
            Calendar cal2 = Calendar.getInstance();
            cal2.setTime(target);

            return isSameDay(cal1, cal2);
        } else {
            return false;
        }
    }

    /**
     * 判断该日期是否是当天日期
     *
     * @param target 目标日期
     * @return true-是当天的日期, 不是当天日期
     */
    public static boolean isToday(Date target) {
        return isSameDay(new Date(), target);
    }

    private static boolean isSameDay(Calendar cal1, Calendar cal2) {
        if (cal1 != null && cal2 != null) {
            return cal1.get(Calendar.ERA) == cal2.get(Calendar.ERA)
                && cal1.get(Calendar.YEAR) == cal2.get(Calendar.YEAR)
                && cal1.get(Calendar.DAY_OF_YEAR) == cal2.get(Calendar.DAY_OF_YEAR);
        } else {
            return false;
        }
    }

}
