package com.bxm.component.auto.registry.spring.cloud;

import com.bxm.component.graceful.shutdown.event.ShutdownEvent;
import com.netflix.appinfo.ApplicationInfoManager;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.shared.Application;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.cloud.netflix.eureka.serviceregistry.EurekaRegistration;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.util.Objects;

/**
 * @author Gonzo
 * @date 2019-12-04 14:56
 */
@Slf4j
public class EurekaClientServiceRegistration {

    private ApplicationContext context;

    private EurekaRegistration registration;

    private ApplicationInfoManager applicationInfoManager;

    private EurekaClientServiceRegistryProperties properties;

    public EurekaClientServiceRegistration(ApplicationContext context, EurekaRegistration registration,
                                           ApplicationInfoManager applicationInfoManager,
                                           EurekaClientServiceRegistryProperties properties) {
        this.context = context;
        this.registration = registration;
        this.applicationInfoManager = applicationInfoManager;
        this.properties = properties;
    }

    /**
     * 当应用完全启动完成，等待处理请求时，将服务置为上线
     *
     * @param event
     */
    @EventListener(ApplicationReadyEvent.class)
    public void onApplicationReady(ApplicationReadyEvent event) {
        if (event.getApplicationContext() == context) {
            log.info("on ApplicationReadyEvent，更新 eureka client 状态为 UP");
            InstanceInfo info = registration.getApplicationInfoManager().getInfo();

            // 由于初始的状态被置为了Starting 所以在默认的启动中，不会去注册服务....
            // 里面的代码可以看到，如果上一个和下一个相同，就不会调用notify()了...
            // 同时，也不可以直接调用 registration.getEurekaClient().setStatus() 方法，在没有注册服务的情况下无法更新状态
            // 默认走一次原本的初始化注册流程 内部会设置info对象的状态，并将信息注册到eureka
            applicationInfoManager.setInstanceStatus(InstanceInfo.InstanceStatus.UP);

            // 有一定几率出现上一次启动没有移除服务，服务还处于OUT_OF_SERVICE状态(kill -9)，因为register行为是无法改变OUT_OF_SERVICE状态的
            // 所以这里需要触发status update to UP
            Application application = registration.getEurekaClient().getApplication(info.getAppName());
            // 如果eureka service 存在当前应用的实例 且实例为OUT状态，则直接置为上线
            if (Objects.nonNull(application)) {
                InstanceInfo remoteInfo = application.getByInstanceId(info.getId());
                if (Objects.nonNull(remoteInfo)
                        && Objects.equals(remoteInfo.getStatus(), InstanceInfo.InstanceStatus.OUT_OF_SERVICE)) {
                    // 服务上线
                    registration.getEurekaClient().setStatus(InstanceInfo.InstanceStatus.UP, info);
                }
            }
        }
    }

    /**
     * 响应自定义的shutdown event
     * 将eureka 服务下线，并阻塞一段时间，避免客户端缓存
     *
     * @param event
     */
    @Order(Ordered.HIGHEST_PRECEDENCE)
    @EventListener(ShutdownEvent.class)
    public void onApplicationEvent(ShutdownEvent event) throws InterruptedException {
        if (event.getApplicationContext() == context) {
            log.info("on ShutdownEvent，更新 eureka client 状态为 OUT_OF_SERVICE");

            try {
                InstanceInfo info = registration.getApplicationInfoManager().getInfo();
                // 服务下线
                registration.getEurekaClient().setStatus(InstanceInfo.InstanceStatus.OUT_OF_SERVICE, info);
                // 阻塞
                Thread.sleep(properties.getAfterOutOfServiceWaitTimeSecond() * 1000);
            } catch (InterruptedException e) {
                log.error("等待 eureka client 下线被中断:{}", e.getMessage(), e);
                throw e;
            }
        }
    }

}
