package com.bxm.component.httpclient.service;

import com.bxm.newidea.component.service.BaseService;
import okhttp3.*;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Objects;

@Service
public class OKHttpService extends BaseService {

    /**
     * 发起get请求
     * @param url 请求地址
     * @return 响应结果
     */
    public String get(String url, Map<String, String> headerMap) {
        return extractResponse(new Request.Builder().url(url).get(), headerMap);
    }

    private String extractResponse(Request.Builder requestBuilder, Map<String, String> headerMap) {
        try {
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                requestBuilder.addHeader(entry.getKey(), entry.getValue());
            }
            Request request = requestBuilder.build();

            OkHttpClient client = new OkHttpClient();
            Response response = client.newCall(request).execute();
            if (response.isSuccessful()) {
                return Objects.requireNonNull(response.body()).string();
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    public String get(String url, Map<String, String> params, Map<String, String> headerMap) {
        Request.Builder request = null;
        try {
            URIBuilder uriBuilder = new URIBuilder(url);
            for (Map.Entry<String, String> entry : params.entrySet()) {
                uriBuilder.addParameter(entry.getKey(), entry.getValue());
            }

            request = new Request.Builder()
                    .url(uriBuilder.toString())
                    .get();

        } catch (URISyntaxException e) {
            logger.error(e.getMessage(), e);
        }
        return extractResponse(request, headerMap);
    }

    /**
     * post提交form表单
     * @param url    提交地址
     * @param params 表单参数KV
     * @return 响应结果
     */
    public String postForm(String url, Map<String, String> params, Map<String, String> headerMap) {
        FormBody.Builder builder = new FormBody.Builder();

        for (Map.Entry<String, String> entry : params.entrySet()) {
            builder.add(entry.getKey(), entry.getValue());
        }

        Request.Builder request = new Request.Builder()
                .url(url)
                .post(builder.build());

        return extractResponse(request, headerMap);
    }

    /**
     * 提交空白的body post请求
     * @param url     请求地址
     * @param content 请求body内容
     * @return 响应结果
     */
    public String postPlainBody(String url, String content, Map<String, String> headerMap) {
        Request.Builder request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(MediaType.parse(String.valueOf(ContentType.TEXT_PLAIN)), content));

        return extractResponse(request, headerMap);
    }

    /**
     * 提交json格式的body请求
     * @param url     请求地址
     * @param content 请求的json格式body
     * @return 响应内容
     */
    public String postJsonBody(String url, String content, Map<String, String> headerMap) {
        Request.Builder request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(MediaType.parse(String.valueOf(ContentType.APPLICATION_JSON)), content));

        return extractResponse(request, headerMap);
    }
}
