package com.bxm.component.mircometer.utils;

import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.util.Collections;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;

/**
 * 添加监控的线程池
 * 做成静态工具类，如果有内部定义的线程池，也可以添加到监控中来
 *
 * @author liujia
 * @date 12/2/21 2:52 PM
 **/
public class TreadPoolMetrics {

    private TreadPoolMetrics() {
    }

    private static final Logger log = LoggerFactory.getLogger(TreadPoolMetrics.class);

    public static void add(String poolName, ThreadPoolTaskExecutor taskExecutor) {
        add(poolName, taskExecutor.getThreadPoolExecutor());
    }

    public static void add(String poolName, ThreadPoolExecutor executor) {
        if (log.isDebugEnabled()) {
            log.debug("初始化线程池[{}]的指标", poolName);
        }

        Iterable<Tag> tags = Collections.singleton(Tag.of("pool.name", poolName));
        // 线程核心线程数量
        Metrics.gauge("thread.pool.core.size", tags, executor, ThreadPoolExecutor::getCorePoolSize);
        // 线程最大值
        Metrics.gauge("thread.pool.largest.size", tags, executor, ThreadPoolExecutor::getLargestPoolSize);
        // 允许创建的线程数量上限
        Metrics.gauge("thread.pool.max.size", tags, executor, ThreadPoolExecutor::getMaximumPoolSize);
        // 当前处于活跃状态的线程数量
        Metrics.gauge("thread.pool.active.size", tags, executor, ThreadPoolExecutor::getActiveCount);
        // 当前线程池中的线程数量
        Metrics.gauge("thread.pool.current.size", tags, executor, ThreadPoolExecutor::getPoolSize);
        // 堆积的任务数量
        Metrics.gauge("thread.pool.queue.size", tags, executor, (targetExecutor) -> {
            BlockingQueue<Runnable> queue = targetExecutor.getQueue();

            int size = queue.size();
            if (size == Integer.MAX_VALUE) {
                size = -1;
            }
            return size;
        });

        if (log.isDebugEnabled()) {
            BlockingQueue<Runnable> queue = executor.getQueue();
            if (null != queue) {
                log.debug("{}使用的队列类型：{}", poolName, queue.getClass().getName());
            }
        }
    }
}
