package com.bxm.listener;

import cn.hutool.core.collection.CollectionUtil;
import com.bxm.handler.NacosConfigChangeHandler;
import com.bxm.newidea.component.tools.StringUtils;
import com.purgeteam.dynamic.config.starter.annotation.EnableDynamicConfigEvent;
import com.purgeteam.dynamic.config.starter.event.ActionConfigEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author lowi
 * @date 2022/3/4 11:37
 */
@Slf4j
@Component
@EnableDynamicConfigEvent
public class NacosConfigListener implements ApplicationListener<ActionConfigEvent>, ApplicationContextAware, ApplicationRunner {

    private Map<String, List<NacosConfigChangeHandler>> modifyConfigHandlerMap = new HashMap<>();

    private static ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        NacosConfigListener.applicationContext = applicationContext;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        Collection<NacosConfigChangeHandler> handlers = applicationContext.getBeansOfType(
                NacosConfigChangeHandler.class)
                .values();
        for (NacosConfigChangeHandler handler : handlers) {
            String listenerKey = handler.listenerConfigKey();
            if (StringUtils.isBlank(listenerKey)) {
                log.error("{} 没有设置需要监听的Key值，监听无效，请配置", handler.getClass().getSimpleName());
                continue;
            }

            listenerKey = minusSignToHump(underlineToHump(listenerKey));

            List<NacosConfigChangeHandler> matchHandlerList = modifyConfigHandlerMap.getOrDefault(
                    listenerKey, new ArrayList<>());
            matchHandlerList.add(handler);
            modifyConfigHandlerMap.put(listenerKey, matchHandlerList);
        }

    }

    @Override
    public void onApplicationEvent(ActionConfigEvent event) {
        Map<String, HashMap> map = event.getPropertyMap();
        for (Map.Entry<String, HashMap> entry : map.entrySet()) {
            String key = entry.getKey();
            String newKey = minusSignToHump(underlineToHump(key));

            //监听到了当前key的配置发生了变化
            List<String> matchKey = new ArrayList<>();

            log.info("监听到nacos配置变更，变动的key:{}", key);
            modifyConfigHandlerMap.forEach((beanConfigKey, bean) -> {
                if (newKey.startsWith(beanConfigKey)) {
                    matchKey.add(beanConfigKey);
                }
            });

            if (CollectionUtil.isEmpty(matchKey)) {
                return;
            }

            for (String listenerKey : matchKey) {
                List<NacosConfigChangeHandler> nacosConfigChangeHandler = modifyConfigHandlerMap.get(listenerKey);

                if (Objects.isNull(nacosConfigChangeHandler)) {
                    return;
                }

                if (Objects.isNull(entry.getValue())) {
                    for (NacosConfigChangeHandler configHandler : nacosConfigChangeHandler) {
                        configHandler.remove(key);
                    }
                    return;
                }

                Map changeMap = entry.getValue();
                log.info("配置[key:{}]被改变，改变前before：{}，改变后after：{}", key, changeMap.get("before"), changeMap.get("after"));

                if (Objects.nonNull(changeMap.get("before")) && Objects.nonNull(changeMap.get("after"))) {
                    for (NacosConfigChangeHandler configHandler : nacosConfigChangeHandler) {
                        configHandler.change(key, changeMap.get("before"), changeMap.get("after"));
                    }
                    return;
                }
            }
        }
    }

    private static Pattern linePattern = Pattern.compile("_(\\w)");

    private static Pattern minusSignPattern = Pattern.compile("-(\\w)");

    /**
     * 下划线转驼峰,正常输出
     */
    public String underlineToHump(String str) {
        Matcher matcher = linePattern.matcher(str);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, matcher.group(1).toUpperCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * 减号-转驼峰,正常输出
     */
    public String minusSignToHump(String str) {
        Matcher matcher = minusSignPattern.matcher(str);
        StringBuffer sb = new StringBuffer();
        while (matcher.find()) {
            matcher.appendReplacement(sb, matcher.group(1).toUpperCase());
        }
        matcher.appendTail(sb);
        return sb.toString();
    }
}
