package com.bxm.component.oncejob.dispatcher;

import com.bxm.component.oncejob.job.OnceJobDefinition;
import com.bxm.component.oncejob.storage.JobRepository;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

/**
 * 一次性定时任务调度器
 *
 * @author liujia
 * @date 7/29/21 4:58 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class OnceJobDispatcher {

    private JobRepository jobRepository;

    /**
     * 推送一个定时任务
     *
     * @param definition 任务定义的相关信息
     * @return 任务ID，如果用户提供了ID，则返回用户定义的ID，否则将创建一个UUID
     */
    public String push(OnceJobDefinition definition) {
        checkBuilder(definition);
        return jobRepository.saveJob(definition);
    }

    /**
     * 取消一个定时任务
     *
     * @param jobId 任务ID，与创建时保持一致
     */
    public void cancel(String jobId) {
        boolean removed = jobRepository.removeJob(jobId);
        if (!removed) {
            log.warn("任务[{}]已执行或不存在，取消失败", jobId);
        }

        if (log.isDebugEnabled()) {
            log.debug("任务[{}]已经删除，删除结果：{}", jobId, removed);
        }
    }

    private void checkBuilder(OnceJobDefinition builder) {
        if (null == builder) {
            throw new IllegalArgumentException("参数OnceJobBuilder不能为空");
        }

        if (builder.getFireDate() == null) {
            throw new IllegalArgumentException("任务执行时间必须提供");
        }

        if (builder.getCallback() == null) {
            throw new IllegalArgumentException("任务回调对象必须提供");
        }
    }
}
