package com.bxm.component.oncejob.storage.memory;

import com.bxm.component.oncejob.converter.JobConverter;
import com.bxm.component.oncejob.job.JobHolder;
import com.bxm.component.oncejob.job.JobPersistentObject;
import com.bxm.component.oncejob.job.OnceJobDefinition;
import com.bxm.component.oncejob.storage.RecentJobRepository;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * @author liujia
 * @date 8/2/21 3:54 PM
 **/
@Slf4j
public class MemoryRecentJobRepositoryImpl implements RecentJobRepository {

    private final List<JobPersistentObject> tempJobList = Collections.synchronizedList(Lists.newArrayList());

    @Override
    public List<JobPersistentObject> load(long lastFireTimeMills, int num) {

        List<JobPersistentObject> result = Lists.newArrayList();
        int fetchNum = num;

        synchronized (tempJobList) {
            Iterator<JobPersistentObject> iterator = tempJobList.iterator();
            while (iterator.hasNext()) {
                JobPersistentObject persistentObject = iterator.next();
                if (fetchNum <= 0) {
                    break;
                }

                if (persistentObject.getFireDate() <= lastFireTimeMills) {
                    result.add(persistentObject);
                    iterator.remove();
                    --fetchNum;
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("剩余任务数量：{}", tempJobList.size());
        }

        return result;
    }

    @Override
    public void pushAll(List<JobPersistentObject> jobs) {
        tempJobList.addAll(jobs);
    }

    @Override
    public String saveJob(OnceJobDefinition definition) {
        JobPersistentObject persistentObject = JobConverter.convert(definition);

        tempJobList.add(persistentObject);

        return persistentObject.getJobId();
    }

    @Override
    public boolean removeJob(String jobId) {
        synchronized (tempJobList) {
            Iterator<JobPersistentObject> iterator = tempJobList.iterator();
            while (iterator.hasNext()) {
                JobPersistentObject persistentObject = iterator.next();
                if (StringUtils.equals(persistentObject.getJobId(), jobId)) {
                    iterator.remove();
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public List<JobPersistentObject> clearDirty() {
        synchronized (tempJobList) {
            Iterator<JobPersistentObject> iterator = tempJobList.iterator();
            while (iterator.hasNext()) {
                JobPersistentObject persistentObject = iterator.next();
                if (persistentObject.getFireDate() < System.currentTimeMillis()) {
                    iterator.remove();

                    if (log.isDebugEnabled()) {
                        log.debug("处理过期任务：{}-{}", persistentObject.getJobId(), persistentObject.getFireDate());
                    }

                    JobHolder.pushJob(persistentObject);
                }
            }
        }

        return Lists.newArrayList();
    }
}
