package com.bxm.component.preheat.config;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.TimeInterval;
import com.bxm.component.preheat.properties.PreheatConfigurationProperties;
import com.bxm.component.preheat.registration.RegistrationNotify;
import com.bxm.newidea.component.core.PreheatPhaser;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.actuate.autoconfigure.availability.AvailabilityHealthContributorAutoConfiguration;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.boot.context.event.EventPublishingRunListener;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import static com.bxm.component.preheat.constant.PreheatConstant.PREHEAT_ENABLED;

/**
 * 监听ApplicationReadyEvent事件，用于调用{@link PreheatPhaser} 执行定义的预热行为
 * 并行执行预热处理器，直到处理完成
 * 处理完成后，触发服务的自动注册，让其他服务或外部流量可以正常调用
 * 因为spring的监听器默认为同步处理，会阻塞后续的处理,优先级需要最低，防止预热处理时，相关基础资源没有初始化
 *
 * 相关class:
 * {@link AvailabilityHealthContributorAutoConfiguration}
 * {@link EventPublishingRunListener#running(org.springframework.context.ConfigurableApplicationContext)}
 *
 * @author liujia
 * @date 2022/07/26 16:18:30
 * @since 2.2.1
 */
@Slf4j
public class PreheatApplicationReadyListener implements ApplicationListener<ApplicationReadyEvent>, Ordered {

    private final List<PreheatPhaser> preheatPhaserList;

    private final List<RegistrationNotify> registrationNotifyList;

    private final PreheatConfigurationProperties properties;

    private ThreadPoolTaskExecutor executor;

    private final AtomicBoolean INIT_FLAG = new AtomicBoolean(false);

    public PreheatApplicationReadyListener(List<PreheatPhaser> preheatPhaserList,
                                           List<RegistrationNotify> registrationNotifyList,
                                           PreheatConfigurationProperties properties) {
        this.preheatPhaserList = preheatPhaserList;
        this.registrationNotifyList = registrationNotifyList;
        this.properties = properties;

        executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(Runtime.getRuntime().availableProcessors() * 2);
        executor.setThreadFactory(new NamedThreadFactory("preheat-phaser-execute"));
        executor.initialize();
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        // 仅执行一次
        if (!INIT_FLAG.compareAndSet(false, true)) {
            return;
        }
        if (CollectionUtil.isEmpty(preheatPhaserList)) {
            return;
        }

        ConfigurableEnvironment environment = event.getApplicationContext().getEnvironment();
        String preheatEnabled = environment.getProperty(PREHEAT_ENABLED, "true");
        if (!Boolean.parseBoolean(preheatEnabled)) {
            log.info("设置了{}为false，跳过预热处理", PREHEAT_ENABLED);
            return;
        }

        TimeInterval timer = DateUtil.timer();
        log.info("[c-preheat] 开始执行预热处理");
        // 执行各个预热周期的执行者
        callPhaser(event);
        // 执行手工注册
        autoRegistrationConfig();
        // 清理资源
        clean();
        log.info("[c-preheat] 预热和注册执行结束，总耗时：{}", timer.intervalMs());
    }

    private void autoRegistrationConfig() {
        log.info("[c-preheat] 开始执行服务注册");
        for (RegistrationNotify registrationNotify : registrationNotifyList) {
            registrationNotify.readinessNotify();
        }
    }

    /**
     * 并行调用spring context中所有的预热执行器，如果调用失败，则终止服务启动
     * 防止不健康的应用被访问
     *
     * @param event 事件
     */
    private void callPhaser(ApplicationReadyEvent event) {
        CountDownLatch countDownLatch = new CountDownLatch(preheatPhaserList.size());
        TimeInterval timer = DateUtil.timer();
        AtomicInteger errorPhaserCount = new AtomicInteger();

        for (PreheatPhaser preheatPhaser : preheatPhaserList) {
            executor.execute(() -> {
                TimeInterval handlerTimer = DateUtil.timer();
                try {
                    preheatPhaser.handle();
                    log.info("[c-preheat] {}预热结束，耗时：{}", preheatPhaser.getClass(), handlerTimer.intervalMs());
                } catch (Exception e) {
                    log.error("[c-preheat] {}预热处理失败", preheatPhaser.getClass());
                    log.error(e.getMessage(), e);
                    errorPhaserCount.incrementAndGet();
                }

                countDownLatch.countDown();
            });
        }

        try {
            if (countDownLatch.await(properties.getExecuteTimeoutMills(), TimeUnit.MILLISECONDS)) {
                log.info("[c-preheat] 预热执行结束，总消耗时间：{}", timer.intervalMs());
            } else {
                log.error("[c-preheat] 执行时间超过{},当前执行时长：{}", properties.getExecuteTimeoutMills(), timer.intervalMs());
            }

            if (errorPhaserCount.get() > 0) {
                log.error("[c-preheat] 应用终止启动，预热过程中发生了异常，请检查");
                ConfigurableApplicationContext applicationContext = event.getApplicationContext();
                applicationContext.stop();
                System.exit(0);
            }
        } catch (InterruptedException e) {
            log.error("[c-preheat] 执行中断," + e.getMessage(), e);
        }
    }

    private void clean() {
        executor.destroy();
        executor = null;
    }

    @Override
    public int getOrder() {
        // 优先级设置为最低，等其他业务处理完成后再处理
        return Ordered.LOWEST_PRECEDENCE;
    }
}





















