package com.bxm.newidea.component.quartz.config;

import org.quartz.Calendar;
import org.quartz.JobDetail;
import org.quartz.Trigger;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.quartz.QuartzProperties;
import org.springframework.boot.autoconfigure.quartz.SchedulerFactoryBeanCustomizer;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;

import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.Executor;

/**
 * 定时器默认spring boot配置
 * 系统中必须进行了quartz相关配置
 * 可以设定一个配置类名称为“custonQuartzConfiguration”的bean来复写配置
 * @author liujia
 */
@Configuration
@EnableScheduling
@EnableConfigurationProperties(QuartzProperties.class)
@ConditionalOnProperty(prefix = "spring.quartz", value = "job-store-type")
@ConditionalOnMissingBean(name = "custonQuartzConfiguration")
public class QuartzConfiguration {

    private final QuartzProperties properties;

    private final List<SchedulerFactoryBeanCustomizer> customizers;

    private final Executor taskExecutor;

    private final JobDetail[] jobDetails;

    private final Map<String, Calendar> calendars;

    private final Trigger[] triggers;

    private final ApplicationContext applicationContext;

    public QuartzConfiguration(QuartzProperties properties,
                               ObjectProvider<List<SchedulerFactoryBeanCustomizer>> customizers,
                               ObjectProvider<Executor> taskExecutor, ObjectProvider<JobDetail[]> jobDetails,
                               ObjectProvider<Map<String, Calendar>> calendars,
                               ObjectProvider<Trigger[]> triggers, ApplicationContext applicationContext) {
        this.properties = properties;
        this.customizers = customizers.getIfAvailable();
        this.taskExecutor = taskExecutor.getIfUnique();
        this.jobDetails = jobDetails.getIfAvailable();
        this.calendars = calendars.getIfAvailable();
        this.triggers = triggers.getIfAvailable();
        this.applicationContext = applicationContext;
    }

    @Bean
    @ConditionalOnMissingBean
    public SchedulerFactoryBean quartzScheduler() {
        SchedulerFactoryBean schedulerFactoryBean = new SchedulerFactoryBean();
        schedulerFactoryBean.setJobFactory(new AutowireCapableBeanJobFactory(
                this.applicationContext.getAutowireCapableBeanFactory()));
        if (!this.properties.getProperties().isEmpty()) {
            schedulerFactoryBean
                    .setQuartzProperties(this.asProperties(this.properties.getProperties()));
        }
        if (this.taskExecutor != null) {
            schedulerFactoryBean.setTaskExecutor(this.taskExecutor);
        }
        if (this.jobDetails != null && this.jobDetails.length > 0) {
            schedulerFactoryBean.setJobDetails(this.jobDetails);
        }
        if (this.calendars != null && !this.calendars.isEmpty()) {
            schedulerFactoryBean.setCalendars(this.calendars);
        }
        if (this.triggers != null && this.triggers.length > 0) {
            schedulerFactoryBean.setTriggers(this.triggers);
        }
        this.customize(schedulerFactoryBean);
        return schedulerFactoryBean;
    }

    private Properties asProperties(Map<String, String> source) {
        Properties properties = new Properties();
        properties.putAll(source);
        return properties;
    }

    private void customize(SchedulerFactoryBean schedulerFactoryBean) {
        if (this.customizers != null) {
            for (SchedulerFactoryBeanCustomizer customizer : this.customizers) {
                customizer.customize(schedulerFactoryBean);
            }
        }
    }

}
