package com.bxm.newidea.component.redis;

import com.fasterxml.jackson.core.type.TypeReference;

import java.util.List;

/**
 * redis 集合操作代理
 * list是有序集合
 * @author liujia 2018/5/22 14:52
 */
public interface RedisListAdapter extends BaseRedisOperation {

    /**
     * 从左边推送values到列表中
     * @param generator key
     * @param values    存储的对象列表,最后一条记录出现在最左侧
     * @return 推入的数量
     */
    Long leftPush(KeyGenerator generator, Object... values);

    /**
     * 从左边推送value到列表中
     * @param generator key
     * @param value     存储的对象列表,最后一条记录出现在最左侧
     * @return 推入的数量
     */
    <T> Long leftPush(KeyGenerator generator, List<T> value);

    /**
     * 从右边推送values到列表中
     * @param generator key
     * @param values    存储的对象列表，最后一条记录出现在最右侧
     * @return 推入的数量
     */
    <T> Long rightPush(KeyGenerator generator, T... values);

    /**
     * 获取当前集合的长度
     * @param generator key
     * @return 长度
     */
    Long size(KeyGenerator generator);

    /**
     * 删除列表左边第一个对象并返回
     * @param generator key
     * @return 删除的对象
     */
    <T> T leftPop(KeyGenerator generator, Class<T> clasz);

    /**
     * 删除列表左边第一个对象并返回,返回对象为泛型
     * @param generator key
     * @return 删除的对象
     */
    <T> T leftPop(KeyGenerator generator, TypeReference<T> typeReference);

    /**
     * 删除列表右边第一个对象并返回
     * @param generator key
     * @return 删除的对象
     */
    <T> T rightPop(KeyGenerator generator, Class<T> clasz);

    /**
     * 删除列表右边第一个对象并返回,返回对象为泛型
     * @param generator key
     * @return 删除的对象
     */
    <T> T rightPop(KeyGenerator generator, TypeReference<T> typeReference);

    /**
     * 返回列表中的指定index的数据
     * @param generator key
     * @param index     列表下标（0开始）
     * @return 返回的对象列表
     */
    <T> List<T> leftIndex(KeyGenerator generator, long index, Class<T> clasz);

    /**
     * 返回列表中的指定index的数据,返回对象为泛型
     * @param generator key
     * @param index     列表下标（0开始）
     * @return 返回的对象列表
     */
    <T> List<T> leftIndex(KeyGenerator generator, long index, TypeReference<T> typeReference);

    /**
     * 返回index对应下标的元素
     * @param generator key
     * @param index     列表下标
     * @return 下标对应的元素，如果不存在返回null
     */
    <T> T index(KeyGenerator generator, long index, Class<T> clasz);

    /**
     * 返回index对应下标的元素,返回对象为泛型
     * @param generator key
     * @param index     列表下标
     * @return 下标对应的元素，如果不存在返回null
     */
    <T> T index(KeyGenerator generator, long index, TypeReference<T> typeReference);

    /**
     * 返回列表中指定范围的数据
     * @param generator key
     * @param start     起始下标（0开始）
     * @param end       结束下标（0开始）
     * @return 返回对象列表
     */
    <T> List<T> range(KeyGenerator generator, long start, long end, Class<T> clasz);

    /**
     * 返回列表中指定范围的数据,返回对象为泛型
     * @param generator key
     * @param start     起始下标（0开始）
     * @param end       结束下标（0开始）
     * @return 返回对象列表
     */
    <T> List<T> range(KeyGenerator generator, long start, long end, TypeReference<T> typeReference);

    /**
     * 列表裁剪，只保存start到end之间的列表内容
     * @param generator key
     * @param start     起始下标（0开始）
     * @param end       结束下标（0开始）
     */
    void leftTrim(KeyGenerator generator, long start, long end);

}
