package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Component;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

@SuppressWarnings("unchecked")
@Component
public class RedisStringAdapterImpl extends BaseRedisAdapter implements RedisStringAdapter {

    private ValueOperations operations;

    @Autowired
    public RedisStringAdapterImpl(RedisTemplate redisTemplate) {
        super(redisTemplate);
        this.operations = redisTemplate.opsForValue();
    }

    @Override
    public Long increment(KeyGenerator generator) {
        return this.increment(generator, 1);
    }

    @Override
    public Long increment(KeyGenerator generator, int amount) {
        return this.incrementWithDefault(generator, 1, amount, 0);
    }

    @Override
    public Long increment(KeyGenerator generator, long expireSeconds) {
        return this.incrementWithDefault(generator, 1, 1, expireSeconds);
    }

    @Override
    public Long incrementWithDefault(KeyGenerator generator, long defaultVal) {
        return this.incrementWithDefault(generator, defaultVal, 1, 0);
    }

    @Override
    public Long incrementWithDefault(KeyGenerator generator, long defaultVal, int amount) {
        return this.incrementWithDefault(generator, defaultVal, amount, 0);
    }

    @Override
    public Long incrementWithDefault(KeyGenerator generator, long defaultVal, int amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, true).longValue();
    }

    @Override
    public Long decrement(KeyGenerator generator, long defaultVal, int amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, false).longValue();
    }

    @Override
    public Long decrement(KeyGenerator generator) {
        return this.decrement(generator, 1, 1, 0);
    }

    @Override
    public Long decrement(KeyGenerator generator, int amount) {
        return this.decrement(generator, 1, amount, 0);
    }

    @Override
    public Long decrement(KeyGenerator generator, long expireSeconds) {
        return this.decrement(generator, 1, 1, expireSeconds);
    }

    @Override
    public Long decrement(KeyGenerator generator, long defaultVal, int amount) {
        return this.decrement(generator, defaultVal, amount, 0);
    }

    @Override
    public Long getLong(KeyGenerator generator) {
        return NumberUtils.parseToLong(this.getString(generator));
    }

    @Override
    public Integer getInt(KeyGenerator generator) {
        return NumberUtils.parseToInt(this.getString(generator));
    }

    @Override
    public String getString(KeyGenerator generator) {
        Object value = this.operations.get(generator.gen());
        if (null == value) {
            return StringUtils.EMPTY;
        }
        Object result = getSerializerWithoutType().deserialize((byte[]) value);
        return result == null ? "" : result.toString();
    }

    @Override
    public <T> T get(KeyGenerator generator, Class<T> clasz) {
        Object value = this.operations.get(generator.gen());
        if (null == value) {
            return null;
        }
        return (T) this.getSerializer(clasz).deserialize((byte[]) value);
    }

    @Override
    public <T> T get(KeyGenerator generator, TypeReference<T> typeReference) {
        Object value = this.operations.get(generator.gen());
        if (null == value) {
            return null;
        }
        return (T) this.getSerializer(typeReference).deserialize((byte[]) value);
    }

    @Override
    public <T> void set(KeyGenerator generator, T value) {
        this.operations.set(generator.gen(), Objects.requireNonNull(this.getSerializerWithoutType().serialize(value)));
    }

    @Override
    public void set(KeyGenerator generator, long value) {
        this.operations.set(generator.gen(), Objects.requireNonNull(this.stringSerializer.serialize(Long.toString(value))));
    }

    @Override
    public <T> void set(KeyGenerator generator, T value, long expireSeconds) {
        this.operations.set(generator.gen(), Objects.requireNonNull(this.getSerializerWithoutType().serialize(value)),
                expireSeconds, TimeUnit.SECONDS);
    }

    @Override
    public void set(KeyGenerator generator, Double value) {
        this.operations.set(generator.gen(), Objects.requireNonNull(this.getSerializerWithoutType().serialize(value)));
    }

    @Override
    public Double increment(KeyGenerator generator, double amount) {
        return this.increment(generator, amount, amount, 0);
    }

    @Override
    public Double increment(KeyGenerator generator, double defaultVal, double amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, true);
    }

    @Override
    public Double decrement(KeyGenerator generator, double amount) {
        return this.decrement(generator, 0, amount, 0);
    }

    @Override
    public Double decrement(KeyGenerator generator, double defaultVal, double amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, false);
    }


    @Override
    public void convertAndSend(String channel, Object message) {
        this.redisTemplate.convertAndSend(channel,Objects.requireNonNull(this.getSerializerWithoutType().serialize(message)));
    }

    private Double exec(KeyGenerator generator, double defaultVal, double amount, long expireSeconds, boolean isIncrement) {
        String key = generator.gen();

        try {
            this.redisTemplate.watch(key);
            this.redisTemplate.multi();

            if (!this.hasKey(generator)) {
                if (expireSeconds > 0) {
                    this.operations.set(key, Objects.requireNonNull(this.stringSerializer.serialize(defaultVal + "")), expireSeconds, TimeUnit.SECONDS);
                } else {
                    this.operations.set(key, Objects.requireNonNull(this.stringSerializer.serialize(defaultVal + "")));
                }
                return defaultVal;
            }
            amount = isIncrement ? amount : -amount;
            return this.operations.increment(key, amount);
        } finally {
            this.redisTemplate.unwatch();
        }
    }
}
