package com.bxm;

import cn.hutool.core.util.RandomUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.JSONObject;
import com.bxm.newidea.component.enums.FileTypeEnum;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.FileUtils;
import com.bxm.newidea.component.vo.ResponseJson;
import com.fasterxml.jackson.core.type.TypeReference;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.http.MediaType;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.ResultActions;
import org.springframework.test.web.servlet.request.MockHttpServletRequestBuilder;
import org.springframework.test.web.servlet.request.MockMultipartHttpServletRequestBuilder;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Date;
import java.util.concurrent.TimeUnit;

import static org.apache.commons.io.FileUtils.openInputStream;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

/**
 * 公共的测试基类
 *
 * @author liujia
 * @date 2022/07/18 09:45:17
 * @since 2.1.21
 */
@ComponentScan("com.bxm")
@RunWith(SpringRunner.class)
@SpringBootTest
@ActiveProfiles("LOCAL")
@AutoConfigureMockMvc
public class BaseTestSupporter {

    @Autowired
    public WebApplicationContext webApplicationContext;

    protected Logger log = LoggerFactory.getLogger(this.getClass());

    @Autowired
    protected MockMvc mockMvc;

    protected <Z> ResponseJson<Z> getJson(MockHttpServletRequestBuilder requestBuilder,
                                          TypeReference<ResponseJson<Z>> typeReference) {
        return getJson(perform(requestBuilder).andReturn(), typeReference);
    }

    protected ResultActions perform(MockHttpServletRequestBuilder request) {
        try {
            return this.mockMvc.perform(request);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    private <Z> ResponseJson<Z> getJson(MvcResult result, TypeReference<ResponseJson<Z>> typeReference) {
        try {
            String content = result.getResponse().getContentAsString(StandardCharsets.UTF_8);
            return JSONObject.parseObject(content, typeReference);
        } catch (UnsupportedEncodingException e) {
            log.error(e.getMessage(), e);
            return null;
        }

    }

    protected void logAndCheck(MockHttpServletRequestBuilder request) {
        logAndCheck(request, "200");
    }

    protected void logAndCheck(MockHttpServletRequestBuilder request, String code) {
        try {
            perform(request)
                .andDo((MvcResult result) -> {
                    String content = result.getResponse().getContentAsString(StandardCharsets.UTF_8);
                    // 用了console的颜色控件,为了醒目一点
                    log.error("request uri:{}", result.getRequest().getRequestURI());
                    log.error("body:{}", content);
                })
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.code").value(code));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    protected <T> MockHttpServletRequestBuilder buildGetRequest(String url, T paramObj) {
        MockHttpServletRequestBuilder request = get(url);
        if (null != paramObj) {
            deepObjBuild(request, null, 0, paramObj);
        }
        return request;
    }

    protected <T> MockHttpServletRequestBuilder buildPostRequest(String url, T paramObj) {
        log.debug("request url:{}", url);

        MockHttpServletRequestBuilder request = post(url);
        request.contentType(MediaType.APPLICATION_JSON);
        request.content(JSON.toJSONString(paramObj));

        return request;
    }

    private void buildArray(MockHttpServletRequestBuilder request, String fieldName, Object array) {
        if (Collection.class.isAssignableFrom(array.getClass())) {
            Collection collectionObj = (Collection) array;
            int index = 0;
            for (Object element : collectionObj) {
                deepObjBuild(request, fieldName, index, element);
                index++;
            }
        }
    }

    private void deepObjBuild(MockHttpServletRequestBuilder request,
                              String parentFieldName,
                              int index,
                              Object childObj) {
        Method[] allDeclaredMethods = ReflectionUtils.getAllDeclaredMethods(childObj.getClass());
        for (Method declaredMethod : allDeclaredMethods) {
            Class<?> returnType = declaredMethod.getReturnType();
            if (!Void.TYPE.isAssignableFrom(returnType)) {
                String fieldName = com.bxm.newidea.component.tools.ReflectionUtils.getFieldName(
                    declaredMethod.getName());

                if (fieldName == null) {
                    continue;
                }

                Object value = ReflectionUtils.invokeMethod(declaredMethod, childObj);

                if (null == value) {
                    continue;
                }

                if (null != parentFieldName) {
                    fieldName = parentFieldName + "[" + index + "]." + fieldName;
                }

                if (MultipartFile.class.isAssignableFrom(returnType)) {
                    if (request instanceof MockMultipartHttpServletRequestBuilder) {
                        ((MockMultipartHttpServletRequestBuilder) request).file((MockMultipartFile) value);
                    } else {
                        log.error(
                            "设置的对象包含文件,但是请求不是MockMultipartHttpServletRequestBuilder的实例," +
                                "使用MockMvcRequestBuilders.multipart构建符合条件的实例");
                    }
                } else if (Date.class.isAssignableFrom(returnType)) {
                    request.param(fieldName, DateUtils.formatDateTime((Date) value));
                } else if (BigDecimal.class.isAssignableFrom(returnType)) {
                    request.param(fieldName, value.toString());
                } else if (com.bxm.newidea.component.tools.ReflectionUtils.isPrimitiveType(returnType)) {
                    request.param(fieldName, value.toString());
                } else if (Collection.class.isAssignableFrom(returnType)) {
                    buildArray(request, fieldName, value);
                } else if (!Class.class.isAssignableFrom(returnType)) {
                    log.info("忽略属性值[{}],其对象类型是[{}]", fieldName, returnType);
                }
            }
        }
    }

    /**
     * 根据test-classes作为根目录,读取filePath对应的文件
     *
     * @param filePath  resource下读取的文件路径,包括文件名
     * @param fieldName 接收的字段名称
     * @return 模拟文件
     */
    protected MockMultipartFile createMockFile(String filePath, String fieldName) {
        String finalFilePath = FileUtils.getClassRootPath() + File.separator + filePath;

        String fileName = FileUtils.getFileName(filePath);
        MockMultipartFile mockFile = null;
        try {
            String contextType = FileTypeEnum.getContentType(FileUtils.getFilextension(fileName));
            mockFile = new MockMultipartFile(
                fieldName,
                fileName,
                contextType,
                openInputStream(new File(finalFilePath))
            );
        } catch (IOException e) {
            log.error("文件不存在:" + finalFilePath, e);
        }
        return mockFile;
    }

    protected String getRandomUrl() {
        return "https://xx.com/" + RandomUtil.randomString(10);
    }

    /**
     * 休眠指定的时间
     *
     * @param seconds 秒
     */
    protected void sleepSeconds(int seconds) {
        try {
            TimeUnit.SECONDS.sleep(seconds);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }
}
