package com.bxm.newidea.component.strategy;

import com.bxm.newidea.component.annotations.StrategyBean;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.OrderComparator;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 策略执行器
 *
 * @author liujia
 * @date 1/13/21 6:03 PM
 **/
@Component
@Slf4j
public class StrategyExecutor implements ApplicationRunner {

    private Map<String, List<IStrategy>> groupStrategyMap = new HashMap<>();

    /**
     * 执行业务逻辑分组内匹配的策略，支持多策略匹配与单策略匹配
     *
     * @param groupName  业务逻辑分组
     * @param context    策略执行上下文
     * @param multiMatch 是否多策略匹配
     */
    @SuppressWarnings("unchecked")
    public <T> void execute(String groupName, T context, boolean multiMatch) {
        List<IStrategy> strategies = groupStrategyMap.get(groupName);

        if (null == strategies) {
            log.error("不存在[{}]对应的策略逻辑分组", groupName);
            return;
        }

        for (IStrategy strategy : strategies) {
            if (strategy.match(context)) {
                strategy.execute(context);

                if (multiMatch) {
                    continue;
                }
                break;
            }
        }

    }

    /**
     * 执行业务逻辑分组内匹配的策略，匹配到第一个策略后就会停止
     *
     * @param groupName 业务分组名称，与{@link IStrategy}实现类上的注解保持一致
     * @param context   策略执行上下文
     */
    public <T> void execute(String groupName, T context) {
        execute(groupName, context, false);
    }

    @Override
    public void run(ApplicationArguments args) {
        for (IStrategy strategy : SpringContextHolder.getBeans(IStrategy.class)) {
            StrategyBean annotation = AnnotationUtils.findAnnotation(strategy.getClass(), StrategyBean.class);
            if (null == annotation) {
                log.warn("[{}]未提供StrategyBean注解", strategy.getClass().getSimpleName());
                continue;
            }

            List<IStrategy> strategies = groupStrategyMap.get(annotation.group());
            if (strategies == null) {
                strategies = new ArrayList<>();
            }
            strategies.add(strategy);
            groupStrategyMap.put(annotation.group(), strategies);
        }

        for (List<IStrategy> strategies : groupStrategyMap.values()) {
            OrderComparator.sort(strategies);
        }
    }
}
