package com.bxm.newidea.component.jwt.tools;

import com.google.common.collect.Maps;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.impl.DefaultClaims;

import java.util.Date;
import java.util.Map;

import static com.bxm.newidea.component.jwt.constant.JwtContant.*;

/**
 * jwt token操作工具
 */
public class JwtTokenUtil {

    private JwtTokenUtil() {
    }

    /**
     * 解析token，从token中获取用户ID
     *
     * @param token 用户当前持有的token
     * @return token中包含的用户ID
     */
    public static String parseToken(String token) {
        String username = null;
        try {
            final Claims claims = getClaimsFromToken(token);
            if (null != claims) {
                username = claims.getSubject();
            }
        } catch (Exception e) {
            // do nothing
        }
        return username;
    }

    private static Date getCreatedDateFromToken(String token) {
        Date created;
        try {
            final Claims claims = getClaimsFromToken(token);
            created = new Date((Long) claims.get(CLAIM_KEY_CREATED));
        } catch (Exception e) {
            created = null;
        }
        return created;
    }

    /**
     * 从token中获取openid
     *
     * @param token token
     * @return openid
     */
    public static String getWxAuthOpenidFromToken(String token) {
        String openid;
        try {
            final Claims claims = getClaimsFromToken(token);
            openid = (String) claims.get(WX_AUTH_OPENID);
        } catch (Exception e) {
            openid = null;
        }
        return openid;
    }

    private static Date getExpirationDateFromToken(String token) {
        Date expiration;
        try {
            final Claims claims = getClaimsFromToken(token);
            expiration = claims.getExpiration();
        } catch (Exception e) {
            expiration = null;
        }
        return expiration;
    }

    /**
     * 解析token，获取存储的相关数据
     *
     * @param token 用户token
     * @return 相关要求数据
     */
    public static Claims getClaimsFromToken(String token) {
        Claims claims;
        try {
            claims = Jwts.parser()
                    .setSigningKey(TOKEN_SECRET)
                    .parseClaimsJws(token)
                    .getBody();
        } catch (Exception e) {
            claims = new DefaultClaims();
        }
        return claims;
    }

    private static Boolean isCreatedBeforeLastPasswordReset(Date created, Date lastPasswordReset) {
        return (lastPasswordReset != null && created.before(lastPasswordReset));
    }

    /**
     * 根据用户ID创建token
     *
     * @param userId 用户ID
     * @return 创建的token值
     */
    public static String generateToken(Long userId) {
        return generateToken(userId, null);
    }

    /**
     * 根据用户ID创建token
     *
     * @param userId 用户ID
     * @return 创建的token值
     */
    public static String generateToken(Long userId, String authCombineCode) {
        Map<String, Object> claims = Maps.newHashMap();
        claims.put(CLAIM_KEY_KEY, userId);
        claims.put(CLAIM_KEY_CREATED, new Date());
        if (null != authCombineCode) {
            claims.put(AUTH_COMBINE_CODE, authCombineCode);
        }
        return generateToken(claims);
    }

    /**
     * 根据openid
     *
     * @param openid 用户openid
     * @return 创建的token值
     */
    public static String generateToken(String openid, Long expiration) {
        Map<String, Object> claims = Maps.newHashMap();
        claims.put(WX_AUTH_OPENID, openid);
        claims.put(CLAIM_KEY_CREATED, new Date());
        return generateToken(claims, expiration);
    }

    private static String generateToken(Map<String, Object> claims) {
        return Jwts.builder()
                .setClaims(claims)
                .setExpiration(generateExpirationDate())
                .signWith(SignatureAlgorithm.HS512, TOKEN_SECRET)
                .compact();
    }

    /**
     * 获取token并且自定义过期时间
     *
     * @param claims     claims
     * @param expiration 过期时间毫秒
     * @return token
     */
    private static String generateToken(Map<String, Object> claims, Long expiration) {
        return Jwts.builder()
                .setClaims(claims)
                .setExpiration(generateExpirationDate(expiration))
                .signWith(SignatureAlgorithm.HS512, TOKEN_SECRET)
                .compact();
    }

    /**
     * token是否可被刷新
     *
     * @param token     当前token
     * @param lastReset 最后重置时间
     * @return true表示可刷新
     */
    public static Boolean canTokenBeRefreshed(String token, Date lastReset) {
        final Date created = getCreatedDateFromToken(token);
        return !isCreatedBeforeLastPasswordReset(created, lastReset) && !isTokenExpired(token);
    }

    /**
     * 更新token中的日期，刷新token
     *
     * @param token 既有token
     * @return 更新后的token
     */
    public static String refreshToken(String token) {
        String refreshedToken;
        try {
            final Claims claims = getClaimsFromToken(token);
            claims.put(CLAIM_KEY_CREATED, new Date());
            refreshedToken = generateToken(claims);
        } catch (Exception e) {
            refreshedToken = null;
        }
        return refreshedToken;
    }

    public static Boolean isTokenExpired(String token) {
        final Date expiration = getExpirationDateFromToken(token);
        if (null == expiration) {
            return true;
        }
        return expiration.before(new Date());
    }

    public static Date generateExpirationDate() {
        long expiration = 72 * 24 * 60 * 60 * 1000L;
        return new Date(System.currentTimeMillis() + expiration);
    }

    public static Date generateExpirationDate(Long expiration) {
        return new Date(System.currentTimeMillis() + expiration);
    }
}