package com.bxm.component.tbk.order.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.api.TbkOrderApi;
import com.bxm.component.tbk.order.config.TbkOpenOrderProperties;
import com.bxm.component.tbk.order.model.dto.TbkOrderInfo;
import com.bxm.component.tbk.order.model.dto.TbkOrderPageInfo;
import com.bxm.component.tbk.order.model.enums.OrderTypeEnum;
import com.bxm.component.tbk.order.model.enums.TbkOrderType;
import com.bxm.component.tbk.order.service.EleTakeOutOrderService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.Objects;

/**
 * 饿了么外卖订单处理
 * @author Gonzo
 * @date  2020-05-03 16:58
 * @since 2.0.0
 */
@Slf4j
public abstract class AbstractEleTakeOutOrderService implements EleTakeOutOrderService {

    @Autowired
    private TbkOrderApi tbOrderService;

    @Autowired
    protected TbkOpenOrderProperties tbkOpenOrderProperties;

    @Override
    public boolean syncOrder(TbkOrderInfo orderInfo) {

        try {
            log.info("处理饿了么订单: {}, siteId: {}订单数据: {} ", orderInfo.getOrderSn(), orderInfo.getSiteId(),
                    JSON.toJSONString(orderInfo));

            return doSyncOrder(orderInfo);
        } catch (Exception e) {
            log.error("处理饿了么订单信息出错 orderInfo: {}", JSON.toJSONString(orderInfo), e);
        }
        return false;
    }

    @Override
    public void syncOrderStatusDays() {
        LocalDateTime now = LocalDateTime.now();
        // 从前3天开始
        LocalDateTime start = now.minusDays(3);

        // 从0点0分开始
        start = LocalDateTime.of(start.getYear(), start.getMonth(), start.getDayOfMonth(),
                0, 0, 0, 0);

        // 前一天 25 59 59结束
        LocalDateTime end = now.minusDays(1L);
        end = LocalDateTime.of(end.getYear(), end.getMonth(), end.getDayOfMonth(),
                23, 59, 59, 999999999);

        // 一个小时一个小时的同步
        while (start.isBefore(end)) {
            syncOrderStatus(start, (start = start.plusHours(1)));
        }

    }

    @Override
    public void syncOrderStatus(LocalDateTime orderStartTime, LocalDateTime orderEndTime) {

        TbkOrderPageInfo<TbkOrderInfo> tbOrderPageInfo;
        Long pageNo = 1L;
        String positionIndex = null;
        do {

            log.info("查询订单信息: orderStartTime :{} orderEndTime: {}, pageNo: {} positionIndex: {}",
                    orderStartTime, orderEndTime, pageNo, positionIndex);

            // 分页查询
            tbOrderPageInfo = tbOrderService.pullOrderNew(pageNo, 20L, positionIndex,
                    orderStartTime, orderEndTime, OrderTypeEnum.CHANNEL_ORDER.getCode(), tbkOpenOrderProperties.getSite());

            if (Objects.isNull(tbOrderPageInfo)) {
                break;
            }

            // 处理订单信息
            if (!CollectionUtils.isEmpty(tbOrderPageInfo.getList())) {
                tbOrderPageInfo.getList().stream()
                        // 过滤出饿了么订单
                        .filter(p -> Objects.equals(p.getTbOrderType(), TbkOrderType.ELE.getType()))
                        // 过滤出当前系统支持的ad zone id
                        .filter(p -> Objects.equals(p.getSiteId(), tbkOpenOrderProperties.getBusinessSiteId()))
                        // 处理饿了么订单
                        .forEach(this::syncOrder);
            }

            positionIndex = tbOrderPageInfo.getPositionIndex();
            pageNo = tbOrderPageInfo.getPageNo();
        } while (tbOrderPageInfo.getHasNext());
    }

    /**
     * 处理饿了么订单信息
     * 不存在入库，存在则更新
     *
     * @param orderInfo
     * @return 处理是否成功
     */
    protected abstract boolean doSyncOrder(TbkOrderInfo orderInfo);

}
